/*
  SamplePluginCpp - An Example KeePass Plugin
  Copyright (C) 2003-2019 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "StdAfx.h"
#include "SamplePluginCpp.h"

using namespace KeePassLib;
using namespace KeePassLib::Security;
using namespace KeePassLib::Utility;

#define MIID_ADDENTRIES _T("SamplePluginCpp_AddEntries")

namespace SamplePluginCpp {

bool SamplePluginCppExt::Initialize(IPluginHost^ host)
{
	if(host == nullptr) return false; // Fail; we need the host
	m_host = host;

	srand(GetTickCount());

	return true; // Initialization successful
}

void SamplePluginCppExt::Terminate()
{
	// Here the plugin should free all resources, close files/streams,
	// remove event handlers, etc.
	// In SamplePluginCpp, there's nothing to clean up.
}

ToolStripMenuItem^ SamplePluginCppExt::GetMenuItem(PluginMenuType t)
{
	// Our menu item below is intended for the main location(s),
	// not for other locations like the group or entry menus
	if(t != PluginMenuType::Main) return nullptr;

	ToolStripMenuItem^ tsmi = gcnew ToolStripMenuItem(_T("SamplePluginCpp"));

	ToolStripMenuItem^ tsmiAddEntries = gcnew ToolStripMenuItem();
	tsmiAddEntries->Text = _T("Add Some Entries");
	tsmiAddEntries->Name = MIID_ADDENTRIES;
	tsmiAddEntries->Click += gcnew EventHandler(this,
		&SamplePluginCppExt::OnMenuAddEntries);
	tsmi->DropDownItems->Add(tsmiAddEntries);

	tsmi->DropDownItems->Add(gcnew ToolStripSeparator());

	ToolStripMenuItem^ tsmiTestNative = gcnew ToolStripMenuItem();
	tsmiTestNative->Text = _T("Test Native C++");
	tsmiTestNative->Click += gcnew EventHandler(this,
		&SamplePluginCppExt::OnMenuTestNative);
	tsmi->DropDownItems->Add(tsmiTestNative);

	// In our handler for the DropDownOpening event, we update the
	// states of our menu items (like disabling the 'Add Some Entries'
	// command when no database is open)
	tsmi->DropDownOpening += gcnew EventHandler(this,
		&SamplePluginCppExt::OnMenuOpening);

	return tsmi;
}

void SamplePluginCppExt::OnMenuOpening(Object^ sender, EventArgs^ e)
{
	UNREFERENCED_PARAMETER(e);

	ToolStripMenuItem^ tsmi = dynamic_cast<ToolStripMenuItem^>(sender);
	if(tsmi == nullptr) { _ASSERT(false); return; }

	array<ToolStripItem^>^ a = tsmi->DropDownItems->Find(MIID_ADDENTRIES, false);
	if((a == nullptr) || (a->Length != 1)) { _ASSERT(false); return; }
	ToolStripItem^ tsiAddEntries = a[0];
	if(tsiAddEntries == nullptr) { _ASSERT(false); return; }

	PwDatabase^ pd = m_host->Database;
	bool bOpen = ((pd != nullptr) && pd->IsOpen);
	tsiAddEntries->Enabled = bOpen;
}

void SamplePluginCppExt::OnMenuAddEntries(Object^ sender, EventArgs^ e)
{
	UNREFERENCED_PARAMETER(sender);
	UNREFERENCED_PARAMETER(e);

	PwDatabase^ pd = m_host->Database;
	if((pd == nullptr) || !pd->IsOpen) { _ASSERT(false); return; }

	PwGroup^ pg = m_host->MainWindow->GetSelectedGroup();
	if(pg == nullptr) pg = pd->RootGroup;

	for(int i = 0; i < 10; ++i)
	{
		PwEntry^ pe = gcnew PwEntry(true, true);

		// Assign a random icon
		pe->IconId = static_cast<PwIcon>(rand() % static_cast<int>(PwIcon::Count));

		// Set some strings of the entry
		pe->Strings->Set(PwDefs::TitleField, gcnew ProtectedString(
			pd->MemoryProtection->ProtectTitle, String::Format(
			_T("Sample Entry #{0}"), i)));
		pe->Strings->Set(PwDefs::PasswordField, gcnew ProtectedString(
			pd->MemoryProtection->ProtectPassword, _T("The secret password")));

		pg->AddEntry(pe, true);
	}

	m_host->MainWindow->UpdateUI(false, nullptr, true, pg, true, nullptr, true);
}

void SamplePluginCppExt::OnMenuTestNative(Object^ sender, EventArgs^ e)
{
	UNREFERENCED_PARAMETER(sender);
	UNREFERENCED_PARAMETER(e);

	// STL can be used
	std::basic_string<TCHAR> strText = _T("This is a native Win32 message box!");

	// Specify the main window as parent for the message box
	HWND hWndMain = reinterpret_cast<HWND>(m_host->MainWindow->Handle.ToPointer());

	// Display a native Win32 message box
	::MessageBox(hWndMain, strText.c_str(), _T("SamplePluginCpp"),
		MB_ICONINFORMATION | MB_OK);
}

} // Namespace SamplePluginCpp
