/*
  KeePass QualityColumn Plugin
  Copyright (C) 2010-2021 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;

using KeePass.Forms;
using KeePass.Plugins;
using KeePass.Resources;
using KeePass.UI;
using KeePass.Util.Spr;

using KeePassLib;
using KeePassLib.Cryptography;

namespace QualityColumn
{
	public sealed class QualityColumnExt : Plugin
	{
		private static IPluginHost g_host = null;
		private QualityColumnProvider m_prov = null;

		internal static IPluginHost Host
		{
			get { return g_host; }
		}

		public override bool Initialize(IPluginHost host)
		{
			Terminate();

			if(host == null) { Debug.Assert(false); return false; }
			g_host = host;

			m_prov = new QualityColumnProvider();
			g_host.ColumnProviderPool.Add(m_prov);

			g_host.MainWindow.FileClosed += this.OnFileClosed;

			return true;
		}

		public override void Terminate()
		{
			if(g_host == null) return;

			g_host.MainWindow.FileClosed -= this.OnFileClosed;

			g_host.ColumnProviderPool.Remove(m_prov);
			m_prov = null;

			g_host = null;
		}

		private void OnFileClosed(object sender, FileClosedEventArgs e)
		{
			QualityColumnProvider.ClearCache();
		}
	}

	public sealed class QualityColumnProvider : ColumnProvider
	{
		private const string QcpName = "Password Quality";

		private static object g_oCacheSync = new object();
		private static Dictionary<string, uint> g_dCache =
			new Dictionary<string, uint>();

		private readonly string[] m_vColNames = new string[] { QcpName };
		public override string[] ColumnNames
		{
			get { return m_vColNames; }
		}

		public override HorizontalAlignment TextAlign
		{
			get { return HorizontalAlignment.Right; }
		}

		internal static void ClearCache()
		{
			lock(g_oCacheSync)
			{
				g_dCache.Clear();
			}
		}

		public override string GetCellData(string strColumnName, PwEntry pe)
		{
			if(strColumnName == null) { Debug.Assert(false); return string.Empty; }
			if(strColumnName != QcpName) return string.Empty;
			if(pe == null) { Debug.Assert(false); return string.Empty; }

			if(!pe.QualityCheck) return KPRes.Disabled;

			string strPw = pe.Strings.ReadSafe(PwDefs.PasswordField);

			if(strPw.IndexOf('{') >= 0)
			{
				IPluginHost host = QualityColumnExt.Host;
				if(host == null) { Debug.Assert(false); return string.Empty; }

				PwDatabase pd = null;
				try
				{
					pd = host.MainWindow.DocumentManager.SafeFindContainerOf(pe);
				}
				catch(Exception) { Debug.Assert(false); }

				SprContext ctx = new SprContext(pe, pd, (SprCompileFlags.Deref |
					SprCompileFlags.TextTransforms), false, false);
				strPw = SprEngine.Compile(strPw, ctx);
			}

			uint uEst;
			lock(g_oCacheSync)
			{
				if(!g_dCache.TryGetValue(strPw, out uEst)) uEst = uint.MaxValue;
			}

			if(uEst == uint.MaxValue)
			{
				uEst = QualityEstimation.EstimatePasswordBits(strPw.ToCharArray());

				lock(g_oCacheSync)
				{
					g_dCache[strPw] = uEst;
				}
			}

			return KPRes.BitsEx.Replace(@"{PARAM}", uEst.ToString());
		}

		// public override bool SupportsCellAction(string strColumnName)
		// {
		//	return true;
		// }

		// public override void PerformCellAction(string strColumnName, PwEntry pe)
		// {
		//	MessageService.ShowInfo(GetCellData(strColumnName, pe));
		// }
	}
}
