﻿/*
  OtpKeyProv Plugin
  Copyright (C) 2011-2015 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;
using System.Diagnostics;

using KeePass.Forms;
using KeePass.Plugins;
using KeePass.Util;

using KeePassLib;
using KeePassLib.Keys;
using KeePassLib.Utility;
using KeePassLib.Serialization;

namespace OtpKeyProv
{
	public sealed class OtpKeyProvExt : Plugin
	{
		private static IPluginHost m_host = null;
		private static OathHotpKeyProv m_prov = null;

		public const string ShortProductName = "OtpKeyProv";
		public const string ProductName = "OtpKeyProv KeePass Plugin";
		public const string HelpFileName = "OtpKeyProv_ReadMe.html";

		private delegate void IocOtpDelegate(IOConnectionInfo ioc, OtpInfo o);

		public static IPluginHost Host
		{
			get { return m_host; }
		}

		public override bool Initialize(IPluginHost host)
		{
			if(m_host != null) { Debug.Assert(false); Terminate(); }
			if(host == null) return false;

			EncodingUtil.SelfTest();

			m_host = host;

			m_prov = new OathHotpKeyProv();
			m_host.KeyProviderPool.Add(m_prov);

			m_host.MainWindow.FileSaved += this.OnFileSaved;
			m_host.MainWindow.FileClosingPost += this.OnFileClosingPost;

			return true;
		}

		public override void Terminate()
		{
			if(m_host == null) return;

			m_host.MainWindow.FileClosingPost -= this.OnFileClosingPost;
			m_host.MainWindow.FileSaved -= this.OnFileSaved;

			m_host.KeyProviderPool.Remove(m_prov);

			m_prov = null;
			m_host = null;
		}

		internal static bool IsHelpPresent()
		{
			try
			{
				string strHelp = UrlUtil.GetFileDirectory(
					WinUtil.GetExecutable(), true, false) +
					HelpFileName;
				return File.Exists(strHelp);
			}
			catch(Exception) { Debug.Assert(false); }

			return false;
		}

		internal static void ConfigureHelpButton(Button btn)
		{
			if(!IsHelpPresent())
			{
				btn.Enabled = false;
				btn.Visible = false;
			}
		}

		internal static void ShowHelp(KeyProviderQueryContext ctx)
		{
			if((ctx != null) && ctx.IsOnSecureDesktop)
			{
				MessageService.ShowWarning("The help file cannot be displayed on the secure desktop.",
					@"Please open the file '" + HelpFileName + @"' manually by double-clicking it in Windows Explorer.");
				return;
			}

			try { Process.Start(HelpFileName); }
			catch(Exception ex) { MessageService.ShowWarning(ex.Message); }
		}

		private void OnFileSaved(object sender, FileSavedEventArgs e)
		{
			if(e == null) { Debug.Assert(false); return; }
			if(!e.Success) return;

			ForEachOtpKey(e.Database, this.UpdateAuxFile);
		}

		private void OnFileClosingPost(object sender, FileClosingEventArgs e)
		{
			if(e == null) { Debug.Assert(false); return; }
			if(e.Cancel) return;

			ForEachOtpKey(e.Database, this.ClearFromCache);
		}

		private void ForEachOtpKey(PwDatabase pd, IocOtpDelegate f)
		{
			if(pd == null) { Debug.Assert(false); return; }
			if(f == null) { Debug.Assert(false); return; }
			if(m_prov == null) { Debug.Assert(false); return; } // For name

			CompositeKey k = pd.MasterKey;
			if(k == null) { Debug.Assert(false); return; }

			foreach(IUserKey uk in k.UserKeys)
			{
				KcpCustomKey ck = (uk as KcpCustomKey);
				if(ck == null) continue;
				if(ck.Name != m_prov.Name) continue;

				byte[] pbKey = ck.KeyData.ReadData();
				if((pbKey == null) || (pbKey.Length == 0)) { Debug.Assert(false); continue; }

				OtpInfo o = OtpInfoCache.GetByHash(pbKey);
				MemUtil.ZeroByteArray(pbKey);
				if(o == null) continue;

				f(pd.IOConnectionInfo, o);
			}
		}

		// Update the aux file, if necessary (e.g. when using
		// 'Save As' to a new file)
		private void UpdateAuxFile(IOConnectionInfo ioc, OtpInfo o)
		{
			if(ioc == null) { Debug.Assert(false); return; }
			if(o == null) { Debug.Assert(false); return; }

			IOConnectionInfo iocAux = OathHotpKeyProv.GetAuxFileIoc(ioc);
			OtpInfo oDisk = OtpInfo.Load(iocAux);

			if((oDisk == null) || (o.Counter > oDisk.Counter))
				OtpInfo.Save(iocAux, o);
		}

		private void ClearFromCache(IOConnectionInfo ioc, OtpInfo o)
		{
			if(o == null) { Debug.Assert(false); return; }

			OtpInfoCache.Remove(o.Secret);
		}
	}
}
