﻿/*
  OtpKeyProv Plugin
  Copyright (C) 2011-2015 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Text;
using System.Security.Cryptography;
using System.Diagnostics;

using KeePassLib.Keys;
using KeePassLib.Utility;

namespace OtpKeyProv
{
	public static class OtpInfoCache
	{
		private static Dictionary<string, OtpInfo> m_dCache =
			new Dictionary<string, OtpInfo>();
		private static List<string> m_lBlocked = new List<string>();

		public static bool Contains(OtpInfo o)
		{
			if(o == null) { Debug.Assert(false); return false; }

			string strHash = Hash(o);
			if(string.IsNullOrEmpty(strHash)) return false;

			return m_dCache.ContainsKey(strHash);
		}

		public static void AddOrBlock(OtpInfo o)
		{
			if(o == null) { Debug.Assert(false); return; }

			string strHash = Hash(o);
			if(string.IsNullOrEmpty(strHash)) { Debug.Assert(false); return; }

			if(m_lBlocked.Contains(strHash)) return;

			if(m_dCache.Remove(strHash)) m_lBlocked.Add(strHash);
			else m_dCache[strHash] = o;
		}

		public static OtpInfo Get(byte[] pbSecret)
		{
			string strHash = Hash(pbSecret);
			if(string.IsNullOrEmpty(strHash)) return null;

			OtpInfo o;
			if(m_dCache.TryGetValue(strHash, out o)) return o;
			return null;
		}

		public static OtpInfo GetByHash(byte[] pbHash)
		{
			foreach(OtpInfo o in m_dCache.Values)
			{
				byte[] pb = o.Secret;
				KcpCustomKey ck = new KcpCustomKey(string.Empty, pb, true);
				if(MemUtil.ArraysEqual(pbHash, ck.KeyData.ReadData()))
					return o;
			}

			return null;
		}

		public static void Remove(byte[] pbSecret)
		{
			string strHash = Hash(pbSecret);
			if(string.IsNullOrEmpty(strHash)) { Debug.Assert(false); return; }

			m_dCache.Remove(strHash);
		}

		private static string Hash(OtpInfo o)
		{
			if(o == null) { Debug.Assert(false); return string.Empty; }

			return Hash(o.Secret);
		}

		private static string Hash(byte[] pbSecret)
		{
			if((pbSecret == null) || (pbSecret.Length == 0))
			{
				Debug.Assert(false);
				return string.Empty;
			}

			SHA256Managed sha = new SHA256Managed();
			byte[] pbHash = sha.ComputeHash(pbSecret);
			sha.Clear();

			return Convert.ToBase64String(pbHash);
		}
	}
}
