﻿/*
  OtpKeyProv Plugin
  Copyright (C) 2011-2015 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.Diagnostics;

using KeePass.UI;

using KeePassLib.Keys;
using KeePassLib.Utility;

namespace OtpKeyProv.Forms
{
	public partial class OtpKeyPromptForm : Form
	{
		private OtpInfo m_otpInfo = null;
		private KeyProviderQueryContext m_kpContext = null;
		private bool m_bRecoveryMode = false;

		private const string CfgAutoAccept = OtpKeyProvExt.ShortProductName +
			"_AutoAccept";
		private const string CfgAutoTab = OtpKeyProvExt.ShortProductName +
			"_AutoTab";

		public void InitEx(OtpInfo otpInfo, KeyProviderQueryContext ctx,
			bool bRecoveryMode)
		{
			m_otpInfo = otpInfo;
			m_kpContext = ctx;
			m_bRecoveryMode = bRecoveryMode;
		}

		public OtpKeyPromptForm()
		{
			InitializeComponent();
		}

		private void OnFormLoad(object sender, EventArgs e)
		{
			if(m_otpInfo == null) { Debug.Assert(false); throw new InvalidOperationException(); }

			GlobalWindowManager.AddWindow(this);

			string strTitle = "Enter OTPs";
			string strDesc = "Unlock using one-time passwords.";

			this.Text = strTitle;
			BannerFactory.CreateBannerEx(this, m_bannerImage,
				OkpRes.B48x48_CryptoChip, strTitle, strDesc);

			OtpKeyProvExt.ConfigureHelpButton(m_btnHelp);

			FontUtil.AssignDefaultBold(m_cbAutoAccept);
			FontUtil.AssignDefaultBold(m_cbAutoTab);

			m_cbAutoAccept.Checked = OtpKeyProvExt.Host.CustomConfig.GetBool(
				CfgAutoAccept, false);
			m_cbAutoTab.Checked = OtpKeyProvExt.Host.CustomConfig.GetBool(
				CfgAutoTab, false);

			foreach(string strSecFmt in EncodingUtil.Formats)
				m_cmbSecretFmt.Items.Add(strSecFmt);
			m_cmbSecretFmt.SelectedIndex = 0;

			TextBox[] vOtpTbs = new TextBox[6] { m_tbOtp1, m_tbOtp2,
				m_tbOtp3, m_tbOtp4, m_tbOtp5, m_tbOtp6 };
			if(m_bRecoveryMode)
			{
				foreach(TextBox tb in vOtpTbs) { tb.Enabled = false; }
				m_lblEnterOtps.Enabled = false;

				m_tabModes.SelectedTab = m_tabRecMode;
			}
			else
			{
				foreach(TextBox tb in vOtpTbs)
				{
					tb.MaxLength = (int)m_otpInfo.OtpLength;
				}

				DisableIfLeq(5, m_tbOtp6);
				DisableIfLeq(4, m_tbOtp5);
				DisableIfLeq(3, m_tbOtp4);
			}
		}

		private void OnFormShown(object sender, EventArgs e)
		{
			if(m_bRecoveryMode) UIUtil.SetFocus(m_tbSecret, this);
			else UIUtil.SetFocus(m_tbOtp1, this);
		}

		private void DisableIfLeq(uint uCount, Control c)
		{
			if(m_otpInfo.OtpsRequired <= uCount)
			{
				c.Enabled = false;
				c.Visible = false;
			}
		}

		private void OnFormClosed(object sender, FormClosedEventArgs e)
		{
			GlobalWindowManager.RemoveWindow(this);
		}

		private void OnBtnOK(object sender, EventArgs e)
		{
			OtpKeyProvExt.Host.CustomConfig.SetBool(CfgAutoAccept,
				m_cbAutoAccept.Checked);
			OtpKeyProvExt.Host.CustomConfig.SetBool(CfgAutoTab,
				m_cbAutoTab.Checked);
			
			try
			{
				OtpDataFmt? fmt = EncodingUtil.GetOtpDataFormat(m_cmbSecretFmt);
				byte[] pbSecret = EncodingUtil.ParseKey(m_tbSecret.Text,
					(fmt.HasValue ? fmt.Value : OtpDataFmt.Hex));
				if(pbSecret != null)
				{
					m_otpInfo.Secret = pbSecret;
					return;
				}

				List<string> lOtps = new List<string>();
				if(m_tbOtp1.TextLength > 0) lOtps.Add(m_tbOtp1.Text);
				if(m_tbOtp2.TextLength > 0) lOtps.Add(m_tbOtp2.Text);
				if(m_tbOtp3.TextLength > 0) lOtps.Add(m_tbOtp3.Text);
				if(m_tbOtp4.TextLength > 0) lOtps.Add(m_tbOtp4.Text);
				if(m_tbOtp5.TextLength > 0) lOtps.Add(m_tbOtp5.Text);
				if(m_tbOtp6.TextLength > 0) lOtps.Add(m_tbOtp6.Text);

				if(!string.IsNullOrEmpty(m_otpInfo.EncryptedSecret)) // < v2.0
				{
					byte[] pbKey32 = OtpUtil.KeyFromOtps(lOtps.ToArray(), 0,
						lOtps.Count, Convert.FromBase64String(
						m_otpInfo.TransformationKey), m_otpInfo.TransformationRounds);
					if(pbKey32 == null) throw new InvalidOperationException();

					pbSecret = OtpUtil.DecryptData(m_otpInfo.EncryptedSecret,
						pbKey32, Convert.FromBase64String(m_otpInfo.EncryptionIV));
					if(pbSecret == null) throw new InvalidOperationException();

					m_otpInfo.Secret = pbSecret;
					m_otpInfo.Counter += (ulong)m_otpInfo.OtpsRequired;
				}
				else // >= v2.0, supporting look-ahead
				{
					bool bSuccess = false;
					for(int i = 0; i < m_otpInfo.EncryptedSecrets.Count; ++i)
					{
						OtpEncryptedData d = m_otpInfo.EncryptedSecrets[i];
						pbSecret = OtpUtil.DecryptSecret(d, lOtps.ToArray(), 0,
							lOtps.Count);
						if(pbSecret != null)
						{
							m_otpInfo.Secret = pbSecret;
							m_otpInfo.Counter += ((ulong)m_otpInfo.OtpsRequired +
								(ulong)i);
							bSuccess = true;
							break;
						}
					}
					if(!bSuccess) throw new InvalidOperationException();
				}
			}
			catch(Exception)
			{
				string strMain = "Failed to create OTP key!";
				string strLine1 = "Make sure you've entered the correct OTPs.";
				string strLine2 = "If you continue to have problems, switch to the 'Recovery mode' tab and provide the OTP generator token's secret key.";

				if(!VistaTaskDialog.ShowMessageBox(strLine1 +
					MessageService.NewParagraph + strLine2, strMain,
					OtpKeyProvExt.ShortProductName, VtdIcon.Warning, this))
					MessageService.ShowWarning(strMain, strLine1, strLine2);

				this.DialogResult = DialogResult.None;
			}
		}

		private void ProcessOtpChanged(uint uChangedID, TextBox tbChanged,
			TextBox tbNext)
		{
			if((tbChanged.TextLength >= (int)m_otpInfo.OtpLength) &&
				m_cbAutoTab.Checked && (tbNext != null) &&
				(uChangedID < m_otpInfo.OtpsRequired))
				UIUtil.SetFocus(tbNext, this);

			if(m_cbAutoAccept.Checked)
			{
				uint r = m_otpInfo.OtpsRequired;

				bool bAll = true;
				if(r >= 1) bAll &= (m_tbOtp1.TextLength == (int)m_otpInfo.OtpLength);
				if(r >= 2) bAll &= (m_tbOtp2.TextLength == (int)m_otpInfo.OtpLength);
				if(r >= 3) bAll &= (m_tbOtp3.TextLength == (int)m_otpInfo.OtpLength);
				if(r >= 4) bAll &= (m_tbOtp4.TextLength == (int)m_otpInfo.OtpLength);
				if(r >= 5) bAll &= (m_tbOtp5.TextLength == (int)m_otpInfo.OtpLength);
				if(r >= 6) bAll &= (m_tbOtp6.TextLength == (int)m_otpInfo.OtpLength);

				if(bAll)
				{
					// Eat last tab that might be sent by the generator
					Application.DoEvents();
					Thread.Sleep(250);
					Application.DoEvents();

					m_btnOK.PerformClick();
				}
			}
		}

		private void OnOtp1TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(1, m_tbOtp1, m_tbOtp2);
		}

		private void OnOtp2TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(2, m_tbOtp2, m_tbOtp3);
		}

		private void OnOtp3TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(3, m_tbOtp3, m_tbOtp4);
		}

		private void OnOtp4TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(4, m_tbOtp4, m_tbOtp5);
		}

		private void OnOtp5TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(5, m_tbOtp5, m_tbOtp6);
		}

		private void OnOtp6TextChanged(object sender, EventArgs e)
		{
			ProcessOtpChanged(6, m_tbOtp6, null);
		}

		private void OnBtnHelp(object sender, EventArgs e)
		{
			OtpKeyProvExt.ShowHelp(m_kpContext);
		}
	}
}
