﻿/*
  On-Screen Keyboard 2 Plugin
  Copyright (C) 2011-2023 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Windows.Forms;

using KeePass;
using KeePass.Forms;
using KeePass.Plugins;
using KeePass.UI;

using KeePassLib;
using KeePassLib.Utility;

namespace OnScreenKeyboard2
{
	public sealed class OnScreenKeyboard2Ext : Plugin
	{
		private IPluginHost m_host = null;

		private Process m_prc = null;
		private bool m_bCloseOsk = false;

		public override bool Initialize(IPluginHost host)
		{
			if(m_host != null) Terminate();
			if(host == null) return false;

			m_host = host;

			GlobalWindowManager.WindowAdded += this.OnGwmWindowAdded;
			GlobalWindowManager.WindowRemoved += this.OnGwmWindowRemoved;

			return true;
		}

		public override void Terminate()
		{
			if(m_host == null) { Debug.Assert(false); return; }

			GlobalWindowManager.WindowAdded -= this.OnGwmWindowAdded;
			GlobalWindowManager.WindowRemoved -= this.OnGwmWindowRemoved;

			m_host = null;
		}

		private void OnGwmWindowAdded(object sender, GwmWindowEventArgs e)
		{
			KeyPromptForm f = ((e != null) ? (e.Form as KeyPromptForm) : null);
			if(f == null) return;

			if(Program.Config.Security.MasterKeyOnSecureDesktop)
			{
				string strMsg = "The on-screen keyboard cannot be shown on the secure desktop.";
				if(!VistaTaskDialog.ShowMessageBox(strMsg, "OnScreenKeyboard2 Plugin",
					PwDefs.ShortProductName, VtdIcon.Warning, f))
					MessageService.ShowWarning(strMsg);
				return;
			}

			try
			{
				string strPath = UrlUtil.EnsureTerminatingSeparator(
					Environment.GetFolderPath(Environment.SpecialFolder.System),
					false) + "Osk.exe";
				if(!File.Exists(strPath)) { Debug.Assert(false); strPath = "Osk.exe"; }

				m_bCloseOsk = (GetOskWindow() == IntPtr.Zero);

				m_prc = Process.Start(strPath);
			}
			catch(Exception) { Debug.Assert(false); }
		}

		private static IntPtr GetOskWindow()
		{
			try { return NativeMethods.FindWindow("OSKMainClass", null); }
			catch(Exception) { Debug.Assert(false); }

			return IntPtr.Zero;
		}

		private void OnGwmWindowRemoved(object sender, GwmWindowEventArgs e)
		{
			KeyPromptForm f = ((e != null) ? (e.Form as KeyPromptForm) : null);
			if(f == null) return;

			if(m_bCloseOsk) { CloseOsk(); m_bCloseOsk = false; }
		}

		private void CloseOsk()
		{
			try
			{
				IntPtr hWnd = GetOskWindow();
				if(hWnd != IntPtr.Zero)
				{
					NativeMethods.SendMessage(hWnd, NativeMethods.WM_CLOSE,
						IntPtr.Zero, IntPtr.Zero);
					Application.DoEvents();
				}
			}
			catch(Exception) { Debug.Assert(false); }

			if(m_prc == null) return;

			try
			{
				if(!m_prc.HasExited)
				{
					m_prc.CloseMainWindow();
					Application.DoEvents();
				}
			}
			catch(Exception) { Debug.Assert(false); }

			try
			{
				if(!m_prc.HasExited)
				{
					m_prc.Kill();
					Application.DoEvents();
				}
			}
			catch(Exception) { Debug.Assert(false); }

			try { m_prc.Close(); }
			catch(Exception) { Debug.Assert(false); }

			m_prc = null;
		}
	}
}
