/*
  KeePass Password Safe - The Open-Source Password Manager
  Copyright (C) 2003-2010 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

using KeePass.DataExchange;
using KeePass.Util;

using KeePassLib;
using KeePassLib.Delegates;
using KeePassLib.Interfaces;
using KeePassLib.Security;
using KeePassLib.Serialization;
using KeePassLib.Utility;

namespace KPScript.ScriptingModules
{
	public sealed class PwDatabaseSaver : IUIOperations
	{
		private PwDatabase m_db;

		public PwDatabaseSaver(PwDatabase pwDb)
		{
			m_db = pwDb;
		}

		public bool UIFileSave(bool bForceSave)
		{
			if(m_db == null) return true;

			try { m_db.Save(null); }
			catch(Exception) { return false; }

			return true;
		}
	}

	public static class DataExchangeMod
	{
		private const string CmdExport = "export";
		private const string CmdSync = "sync";

		public static bool ProcessCommand(string strCommand, CommandLineArgs args,
			PwDatabase pwDatabase, out bool bNeedsSave)
		{
			bNeedsSave = false;

			if(strCommand == CmdExport)
				PerformExport(pwDatabase, args);
			else if(strCommand == CmdSync)
				PerformSync(pwDatabase, args); // No save on sync
			else return false;

			return true;
		}

		private static void PerformExport(PwDatabase pwDb, CommandLineArgs args)
		{
			string strFormat = args["format"];
			if((strFormat == null) || (strFormat.Length == 0))
			{
				KPScript.Program.WriteLineColored("E: Invalid format!", ConsoleColor.Red);
				return;
			}

			FileFormatPool ffp = KeePass.Program.FileFormatPool;
			FileFormatProvider prov = ffp.Find(strFormat);
			if(prov == null)
			{
				KPScript.Program.WriteLineColored("E: Unknown format!", ConsoleColor.Red);
				return;
			}

			if(prov.SupportsExport == false)
			{
				KPScript.Program.WriteLineColored("E: No export support for this format!",
					ConsoleColor.Red);
				return;
			}

			if(prov.TryBeginExport() == false)
			{
				KPScript.Program.WriteLineColored("E: Format initialization failed!",
					ConsoleColor.Red);
				return;
			}

			FileStream fs;
			try
			{
				fs = new FileStream(args["outfile"], FileMode.Create, FileAccess.Write,
					FileShare.None);
			}
			catch(Exception exFs)
			{
				KPScript.Program.WriteLineColored("E: " + exFs.Message, ConsoleColor.Red);
				return;
			}

			PwExportInfo pwInfo = new PwExportInfo(pwDb.RootGroup, pwDb, true);

			if(prov.Export(pwInfo, fs, null))
				KPScript.Program.WriteLineColored("OK: Export succeeded!", ConsoleColor.Green);
			else
				KPScript.Program.WriteLineColored("E: Export failed!", ConsoleColor.Red);
		}

		private static void PerformSync(PwDatabase pwDb, CommandLineArgs args)
		{
			string strFile = args["file"];
			if(string.IsNullOrEmpty(strFile))
			{
				KPScript.Program.WriteLineColored("E: No file specified to synchronize with!", ConsoleColor.Red);
				return;
			}

			IOConnectionInfo ioc = IOConnectionInfo.FromPath(strFile);

			bool? b = ImportUtil.Synchronize(pwDb, new PwDatabaseSaver(pwDb), ioc, true, null);
			if(b.HasValue && b.Value)
				KPScript.Program.WriteLineColored("OK: Synchronization succeeded!", ConsoleColor.Green);
			else if(b.HasValue)
				KPScript.Program.WriteLineColored("E: Synchronization failed!", ConsoleColor.Red);
		}
	}
}
