/*
  KeeAutoExec Plugin
  Copyright (C) 2009-2022 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using KeePass.DataExchange;
using KeePass.Forms;
using KeePass.Plugins;
using KeePass.Resources;
using KeePass.UI;
using KeePass.Util;
using KeePass.Util.Spr;

using KeePassLib;
using KeePassLib.Keys;
using KeePassLib.Security;
using KeePassLib.Serialization;
using KeePassLib.Utility;

namespace KeeAutoExec
{
	public sealed class KeeAutoExecExt : Plugin
	{
		[Flags]
		private enum KaeStrOpt
		{
			None = 0,
			Trim = 1,
			Unquote = 2, // UrlUtil.GetQuotedAppPath

			Path = (Trim | Unquote)
		}

		private IPluginHost m_host = null;
		private ToolStripMenuItem m_tsmiFile = null;
		private ToolStripMenuItem m_tsmiOpen = null;
		private DynamicMenu m_dynOpen = null;

		public override bool Initialize(IPluginHost host)
		{
			Terminate();

			if(host == null) return false;
			m_host = host;

			m_tsmiFile = (m_host.MainWindow.MainMenu.Items[0] as ToolStripMenuItem);
			if(m_tsmiFile == null)
			{
				Debug.Assert(false);
				m_tsmiFile = m_host.MainWindow.ToolsMenu;
			}

			m_tsmiOpen = new ToolStripMenuItem(StrUtil.RemoveAccelerator(
				KPRes.OpenCmd) + @" (&KeeAutoExec)");
			m_tsmiOpen.DropDownOpening += this.OnMenuOpenOpening;
			if(m_tsmiFile.DropDownItems.Count >= 3)
				m_tsmiFile.DropDownItems.Insert(3, m_tsmiOpen);
			else
			{
				Debug.Assert(false);
				m_tsmiFile.DropDownItems.Add(m_tsmiOpen);
			}

			m_dynOpen = new DynamicMenu(m_tsmiOpen);
			m_dynOpen.MenuClick += this.OnMenuOpenClick;

			m_host.MainWindow.FileOpened += this.OnFileOpen;

			UpdateMenus(); // Ensure the popup arrow is displayed

			return base.Initialize(host);
		}

		public override void Terminate()
		{
			if(m_host == null) return;

			m_host.MainWindow.FileOpened -= this.OnFileOpen;

			m_dynOpen.Clear();
			m_dynOpen.MenuClick -= this.OnMenuOpenClick;
			m_dynOpen = null;

			m_tsmiOpen.DropDownOpening -= this.OnMenuOpenOpening;
			m_tsmiFile.DropDownItems.Remove(m_tsmiOpen);
			m_tsmiOpen = null;

			m_tsmiFile = null;
			m_host = null;
			base.Terminate();
		}

		private static int PrioritySort(AutoExecItem x, AutoExecItem y)
		{
			if(x == null) { Debug.Assert(false); return ((y == null) ? 0 : -1); }
			if(y == null) { Debug.Assert(false); return 1; }

			return x.Priority.CompareTo(y.Priority);
		}

		private static void AddAutoExecEntries(List<PwEntry> l, PwGroup pg)
		{
			if(pg.Name.Equals("AutoOpen", StrUtil.CaseIgnoreCmp))
				l.AddRange(pg.GetEntries(true));
			else
			{
				foreach(PwGroup pgSub in pg.Groups)
					AddAutoExecEntries(l, pgSub);
			}
		}

		private static List<AutoExecItem> GetAutoExecItems(PwDatabase pd)
		{
			List<AutoExecItem> l = new List<AutoExecItem>();
			if(pd == null) { Debug.Assert(false); return l; }
			if(!pd.IsOpen) return l;

			PwGroup pgRoot = pd.RootGroup;
			if(pgRoot == null) { Debug.Assert(false); return l; }

			List<PwEntry> lAutoEntries = new List<PwEntry>();
			AddAutoExecEntries(lAutoEntries, pgRoot);

			long lPriStd = 0;
			foreach(PwEntry pe in lAutoEntries)
			{
				string str = pe.Strings.ReadSafe(PwDefs.UrlField);
				if(str.Length == 0) continue;

				AutoExecItem a = new AutoExecItem(pe, pd);
				l.Add(a);

				SprContext ctx = new SprContext(pe, pd, SprCompileFlags.All);

				if(pe.Expires && (pe.ExpiryTime <= DateTime.UtcNow))
					a.Enabled = false;

				bool? ob = GetBoolEx(pe, "Enabled", ctx);
				if(ob.HasValue) a.Enabled = ob.Value;

				ob = GetBoolEx(pe, "Visible", ctx);
				if(ob.HasValue) a.Visible = ob.Value;

				long lItemPri = lPriStd;
				if(GetString(pe, "Priority", ctx, out str))
					long.TryParse(str, out lItemPri);
				a.Priority = lItemPri;

				if(GetString(pe, "IfDevice", ctx, out str))
					a.IfDevice = str;

				++lPriStd;
			}

			l.Sort(KeeAutoExecExt.PrioritySort);
			return l;
		}

		private void OnFileOpen(object sender, FileOpenedEventArgs e)
		{
			if(e == null) { Debug.Assert(false); return; }

			string strDevice = null;
			try { strDevice = Environment.MachineName.Trim(); }
			catch(Exception) { Debug.Assert(false); }

			List<AutoExecItem> l = GetAutoExecItems(e.Database);
			foreach(AutoExecItem a in l)
			{
				if(!a.Enabled) continue;
				if(!IsDeviceEnabled(a, strDevice)) continue;

				try { AutoOpenEntryPriv(a, false); }
				catch(Exception ex)
				{
					MessageService.ShowWarning(ex);
				}
			}
		}

		private void AutoOpenEntryPriv(AutoExecItem a, bool bManual)
		{
			if(m_host == null) { Debug.Assert(false); return; }

			PwEntry pe = a.Entry;
			PwDatabase pdContext = a.Database;

			SprContext ctxNoEsc = new SprContext(pe, pdContext, SprCompileFlags.All);
			SprContext ctxEsc = new SprContext(pe, pdContext, SprCompileFlags.All,
				false, true);

			string strDb;
			if(!GetString(pe, PwDefs.UrlField, ctxEsc, KaeStrOpt.Path, out strDb)) return;

			IOConnectionInfo ioc = IOConnectionInfo.FromPath(strDb);
			if(ioc.IsLocalFile() && !UrlUtil.IsAbsolutePath(strDb))
				ioc = IOConnectionInfo.FromPath(UrlUtil.MakeAbsolutePath(
					WinUtil.GetExecutable(), strDb));
			if(ioc.Path.Length == 0) return;

			string strIocUserName;
			if(GetString(pe, "IocUserName", ctxNoEsc, out strIocUserName))
				ioc.UserName = strIocUserName;

			string strIocPassword;
			if(GetString(pe, "IocPassword", ctxNoEsc, out strIocPassword))
				ioc.Password = strIocPassword;

			if((strIocUserName.Length != 0) && (strIocPassword.Length != 0))
				ioc.IsComplete = true;

			string str;
			if(GetString(pe, "IocProperties", ctxNoEsc, out str))
			{
				IocProperties p = IocProperties.Deserialize(str);
				if(p != null) p.CopyTo(ioc.Properties);
			}

			if(GetString(pe, "IocTimeout", ctxNoEsc, out str))
			{
				long l;
				if(long.TryParse(str, out l))
					ioc.Properties.SetLong(IocKnownProperties.Timeout, l);
			}

			bool? ob = GetBoolEx(pe, "IocPreAuth", ctxNoEsc);
			if(ob.HasValue)
				ioc.Properties.SetBool(IocKnownProperties.PreAuth, ob.Value);

			if(GetString(pe, "IocUserAgent", ctxNoEsc, out str))
				ioc.Properties.Set(IocKnownProperties.UserAgent, str);

			ob = GetBoolEx(pe, "IocExpect100Continue", ctxNoEsc);
			if(ob.HasValue)
				ioc.Properties.SetBool(IocKnownProperties.Expect100Continue, ob.Value);

			ob = GetBoolEx(pe, "IocPassive", ctxNoEsc);
			if(ob.HasValue)
				ioc.Properties.SetBool(IocKnownProperties.Passive, ob.Value);

			ob = GetBoolEx(pe, "SkipIfNotExists", ctxNoEsc);
			if(!ob.HasValue) // Backw. compat.
				ob = GetBoolEx(pe, "Skip if not exists", ctxNoEsc);
			if(ob.HasValue && ob.Value)
			{
				if(!IOConnection.FileExists(ioc)) return;
			}

			CompositeKey ck = new CompositeKey();

			if(GetString(pe, PwDefs.PasswordField, ctxNoEsc, KaeStrOpt.None, out str))
				ck.AddUserKey(new KcpPassword(str));

			if(GetString(pe, PwDefs.UserNameField, ctxNoEsc, KaeStrOpt.Path, out str))
			{
				string strAbs = str;
				IOConnectionInfo iocKey = IOConnectionInfo.FromPath(strAbs);
				if(iocKey.IsLocalFile() && !UrlUtil.IsAbsolutePath(strAbs))
					strAbs = UrlUtil.MakeAbsolutePath(WinUtil.GetExecutable(), strAbs);

				ob = GetBoolEx(pe, "SkipIfKeyFileNotExists", ctxNoEsc);
				if(ob.HasValue && ob.Value)
				{
					IOConnectionInfo iocKeyAbs = IOConnectionInfo.FromPath(strAbs);
					if(!IOConnection.FileExists(iocKeyAbs)) return;
				}

				try { ck.AddUserKey(new KcpKeyFile(strAbs)); }
				catch(InvalidOperationException)
				{
					throw new Exception(strAbs + MessageService.NewParagraph +
						KPRes.KeyFileError);
				}
				catch(Exception) { throw; }
			}
			else // Try getting key file from attachments
			{
				ProtectedBinary pBin = pe.Binaries.Get("KeyFile.bin");
				if(pBin != null)
					ck.AddUserKey(new KcpKeyFile(IOConnectionInfo.FromPath(
						StrUtil.DataToDataUri(pBin.ReadData(), null))));
			}

			if(GetString(pe, "KeyProvider", ctxNoEsc, out str))
			{
				KeyProvider kp = m_host.KeyProviderPool.Get(str);
				if(kp == null)
					throw new Exception(@"Unknown key provider: '" + str + @"'!");

				KeyProviderQueryContext ctxKP = new KeyProviderQueryContext(
					ioc, false, false);

				bool bPerformHash = !kp.DirectKey;
				byte[] pbProvKey = kp.GetKey(ctxKP);
				if((pbProvKey != null) && (pbProvKey.Length != 0))
				{
					ck.AddUserKey(new KcpCustomKey(str, pbProvKey, bPerformHash));
					MemUtil.ZeroByteArray(pbProvKey);
				}
				else return; // Provider has shown error message
			}

			ob = GetBoolEx(pe, "UserAccount", ctxNoEsc);
			if(ob.HasValue && ob.Value)
				ck.AddUserKey(new KcpUserAccount());

			if(ck.UserKeyCount == 0) return;

			GetString(pe, "Focus", ctxNoEsc, out str);
			bool bRestoreFocus = str.Equals("Restore", StrUtil.CaseIgnoreCmp);
			PwDatabase pdPrev = m_host.MainWindow.ActiveDatabase;

			m_host.MainWindow.OpenDatabase(ioc, ck, true);

			if(bRestoreFocus && (pdPrev != null) && !bManual)
			{
				PwDocument docPrev = m_host.MainWindow.DocumentManager.FindDocument(
					pdPrev);
				if(docPrev != null) m_host.MainWindow.MakeDocumentActive(docPrev);
				else { Debug.Assert(false); }
			}
		}

		private void UpdateMenus()
		{
			if(m_host == null) { Debug.Assert(false); return; }

			DynamicMenu d = m_dynOpen;
			if(d == null) { Debug.Assert(false); return; }

			d.Clear();

			int nItems = 0;
			bool bAddSepNext = false;
			foreach(PwDocument doc in m_host.MainWindow.DocumentManager.Documents)
			{
				List<AutoExecItem> l = GetAutoExecItems(doc.Database);
				foreach(AutoExecItem a in l)
				{
					if(!a.Visible) continue;

					SprContext ctx = new SprContext(a.Entry, a.Database,
						SprCompileFlags.NonActive & ~(SprCompileFlags.DateTime |
						SprCompileFlags.EnvVars), false, true);
					string strUrl;
					GetString(a.Entry, PwDefs.UrlField, ctx, KaeStrOpt.Path, out strUrl);

					string strText = (nItems + 1).ToString();
					if((nItems + 1) <= 9) strText = @"&" + strText;
					strText += " " + strUrl;

					if(bAddSepNext)
					{
						d.AddSeparator();
						bAddSepNext = false;
					}

					ToolStripMenuItem tsmi = d.AddItem(strText, null, a);
					++nItems;

					if(!a.Enabled) tsmi.ForeColor = Color.DarkRed;
				}

				bAddSepNext = true;
			}

			if(nItems == 0)
			{
				ToolStripMenuItem tsmi = d.AddItem("(" + KPRes.None + ")", null);
				tsmi.Enabled = false;
			}
		}

		private void OnMenuOpenOpening(object sender, EventArgs e)
		{
			UpdateMenus();
		}

		private void OnMenuOpenClick(object sender, DynamicMenuEventArgs e)
		{
			if(e == null) { Debug.Assert(false); return; }

			try
			{
				AutoExecItem a = (e.Tag as AutoExecItem);
				if(a == null) { Debug.Assert(false); return; }

				AutoOpenEntryPriv(a, true);
			}
			catch(Exception ex)
			{
				MessageService.ShowWarning(ex);
			}
		}

		private static bool GetString(PwEntry pe, string strName, SprContext ctx,
			out string strValue)
		{
			return GetString(pe, strName, ctx, KaeStrOpt.Trim, out strValue);
		}

		private static bool GetString(PwEntry pe, string strName, SprContext ctx,
			KaeStrOpt so, out string strValue)
		{
			if((pe == null) || string.IsNullOrEmpty(strName))
			{
				Debug.Assert(false);
				strValue = string.Empty;
				return false;
			}

			string str = pe.Strings.ReadSafe(strName);
			if(ctx != null) str = SprEngine.Compile(str, ctx);

			if((so & KaeStrOpt.Unquote) != KaeStrOpt.None)
				str = UrlUtil.GetQuotedAppPath(str);
			if((so & KaeStrOpt.Trim) != KaeStrOpt.None)
				str = str.Trim();

			strValue = str;
			return (str.Length != 0);
		}

		private static bool? GetBoolEx(PwEntry pe, string strName, SprContext ctx)
		{
			string str;
			if(GetString(pe, strName, ctx, out str))
			{
				if(str.Equals("True", StrUtil.CaseIgnoreCmp))
					return true;
				if(str.Equals("False", StrUtil.CaseIgnoreCmp))
					return false;
			}

			return null;
		}

		private static bool IsDeviceEnabled(AutoExecItem a, string strDevice)
		{
			string strList = a.IfDevice;
			if(string.IsNullOrEmpty(strList) || string.IsNullOrEmpty(strDevice))
				return true;

			CsvOptions opt = new CsvOptions();
			opt.BackslashIsEscape = false;
			opt.TrimFields = true;

			CsvStreamReaderEx csv = new CsvStreamReaderEx(strList, opt);
			string[] vFlt = csv.ReadLine();
			if(vFlt == null) { Debug.Assert(false); return true; }

			bool bHasIncl = false, bHasExcl = false;
			foreach(string strFlt in vFlt)
			{
				if(string.IsNullOrEmpty(strFlt)) continue;

				if(strFlt[0] == '!') // Exclusion
				{
					if(strDevice.Equals(strFlt.Substring(1), StrUtil.CaseIgnoreCmp))
						return false;
					bHasExcl = true;
				}
				else // Inclusion
				{
					if(strDevice.Equals(strFlt, StrUtil.CaseIgnoreCmp))
						return true;
					bHasIncl = true;
				}
			}

			return (bHasExcl || !bHasIncl);
		}
	}
}
