﻿using System;
using System.Collections.Generic;
using System.Text;
using KeePass.DataExchange;
using KeePass.Resources;
using KeePassLib;
using System.IO;
using KeePassLib.Interfaces;
using System.Linq;
using System.Xml.Linq;
using System.Xml;
using System.Globalization;
using KeePassLib.Utility;
using KeePassLib.Security;

namespace eWalletLiberatedDataImporter
{
	public sealed class Importer : FileFormatProvider
	{
		public override string ApplicationGroup { get { return KPRes.PasswordManagers; } }
		public override string DefaultExtension { get { return "xml"; } }
		public override string FormatName { get { return "eWallet Liberated Data"; } }
		public override System.Drawing.Image SmallIcon { get { return Properties.Resources.eWallet.ToBitmap(); } }
		
		public override bool SupportsExport { get { return false; } }
		public override bool SupportsImport { get { return true; } }
		public override bool SupportsUuids { get { return true; } }

		//TODO: Replace this with something less hacky. Concurrent imports don't currently happen, but there's no guarantee of that.
		private PwDatabase mCurrentImportTarget;

		public override void Import(PwDatabase pwStorage, Stream sInput, IStatusLogger slLogger)
		{
			mCurrentImportTarget = pwStorage;

			XElement wallet = XElement.Load(XmlReader.Create(sInput));
			foreach (var category in wallet.Elements("Category")) //eWallet top level supports only categories - all cards belong to a category
			{
				var group = ImportCategory(category);
				pwStorage.RootGroup.AddGroup(group, true);
			}

			mCurrentImportTarget = null;
		}

		/// <summary>
		/// Create a PwGroup from importing a Category
		/// </summary>
		private PwGroup ImportCategory(XElement category)
		{
			//Create the group for the category
			var group = new PwGroup()
			{
				Uuid = ImportId(category.Attribute("id")),
				Name = (string)category.Attribute("name"),
				IconId = ImportIcon(category.Attribute("standardIcon"))
				//TODO: Support for custom icons?
			};
			
			//Import any subcategories of this category
			foreach (var subCategory in category.Elements("Category"))
			{
				group.AddGroup(ImportCategory(subCategory), true);
			}

			//Import any cards in this category
			foreach (var card in category.Elements("Card"))
			{
				group.AddEntry(ImportCard(card), true);
			}

			return group;
		}

		private PwEntry ImportCard(XElement card)
		{
			var entry = new PwEntry(false, false)
			{
				Uuid = ImportId(card.Attribute("id")),
				IconId = ImportIcon(card.Attribute("standardIcon"))
				//TODO: Support for custom icons?
			};
			var backgroundColor = (string)card.Attribute("backgroundColor");
			if (backgroundColor != null) entry.BackgroundColor = System.Drawing.ColorTranslator.FromHtml(backgroundColor);

			var textColor = (string)card.Attribute("textColor");
			if (textColor != null) entry.ForegroundColor = System.Drawing.ColorTranslator.FromHtml(textColor);

			//Standard fields
			SetField(entry, PwDefs.TitleField,		(string)card.Attribute("name"),				mCurrentImportTarget.MemoryProtection.ProtectTitle);
			SetField(entry, PwDefs.UserNameField,	GetFieldOfKind(card, "AutoPassUser"),		mCurrentImportTarget.MemoryProtection.ProtectUserName);
			SetField(entry, PwDefs.PasswordField,	GetFieldOfKind(card, "AutoPassPassword"),	mCurrentImportTarget.MemoryProtection.ProtectPassword);
			SetField(entry, PwDefs.UrlField,		GetFieldOfKind(card, "AutoPassUrl"),		mCurrentImportTarget.MemoryProtection.ProtectUrl);
			SetField(entry, PwDefs.NotesField,		card.Element("Notes").Value,				mCurrentImportTarget.MemoryProtection.ProtectNotes);
			
			//Custom fields
			foreach(XElement field in from field in card.Descendants("Field")
													let kind = (string)field.Attribute("kind")
															where	!field.IsEmpty &&
																	//Don't include fields that have already been processed as special
																	kind != "AutoPassUser" &&
																	kind != "AutoPassPassword" &&
																	kind != "AutoPassUrl" &&
																	kind != "Notes" &&
																	//eWallet doesn't show the card title on the card, so often there is another property that just repeats this. Don't include that.
																	field.Value != (string)card.Attribute("name")
																select field)
			{
				//Fields of type "Hidden" are protected, all others are not.
				var label = (string)field.Attribute("label");
				if(PwDefs.IsStandardField(label))
				{
					//Can't add custom fields with the same name as standard fields, so prefix the field name with _
					//TODO: Can anything nicer be done here?
					label = "_" + label;
				}
				SetField(entry, label, field.Value, (string)field.Attribute("kind") == "Hidden");
			}

			var image = card.Element("Image");
			if (image != null)
			{
				AddBinary(entry, image);
			}
			var attachment = card.Element("Attachment");
			if (attachment != null)
			{
				AddBinary(entry, attachment);
			}

			return entry;
		}

		
		private void SetField(PwEntry entry, string field, string value, bool protect)
		{
			if (value != null)
			{
				entry.Strings.Set(field, new ProtectedString(protect, value));
			}
		}

		private string GetFieldOfKind(XElement card, string kind)
		{
			return (from field in card.Descendants("Field") where (string)field.Attribute("kind") == kind select field.Value).FirstOrDefault();
		}

		private void AddBinary(PwEntry entry, XElement binary)
		{
			if (!binary.IsEmpty) //KeePass doesn't support referenced files, so only add if data is embedded. TODO: Auto-import referenced files?
			{
				//NOTE: Why does UrlUtil.GetFileName exist - what's wrong with System.IO.Path.GetFilename? Using UrlUtil just in case there's a good reason.
				//UsingProtection always false, eWallet has no concept of it.
				entry.Binaries.Set(UrlUtil.GetFileName((string)binary.Attribute("path")), new ProtectedBinary(false, Convert.FromBase64String(binary.Value)));
			}
		}

		private PwIcon ImportIcon(XAttribute icon)
		{
			//Look up appropriate KeePass icon for standard eWallet icons
			switch ((string)icon)
			{
				case "Card":
				case "CreditCard":
				case "Money":
				case "Visa":
				case "MasterCard":
				case "Financial":
				case "AmericanExpress":
				case "Euro":
				case "Pound":
					return PwIcon.Money;
				case "Notes":
				case "FreeFormInfo":
					return PwIcon.Note;
				case "Person":
					return PwIcon.UserKey;
				case "Keys":
					return PwIcon.Key;
				case "SocialSecurity":
				case "Passport":
				case "DriversLicense":
				case "Membership":
				case "Voter":
				case "Travel":
				case "Identification":
					return PwIcon.Identity;
				case "Library":
					return PwIcon.Book;
				case "CallingCard":
					return PwIcon.UserCommunication;
				case "SerialNumber":
				case "Software":
					return PwIcon.CDRom;
				case "Picture":
					return PwIcon.Thumbnail;
				case "Important":
				case "Star":
					return PwIcon.Star;
				case "Buildings":
				case "House":
					return PwIcon.Home;
				case "World":
					return PwIcon.World;
				case "Lock":
					return PwIcon.LockOpen;
				case "Computer":
					return PwIcon.WorldComputer;
				case "Mobile":
				case "IPhone":
				case "IPod":
					return PwIcon.BlackBerry;
				
				case "GeneralPurpose":
				case "Clothes":
				case "Telephone":
				case "Umbrella":
				case "Health":
				case "Auto":
				case "Category":
				case "Government":
				case "Desk":
				case "Lens":
				case "Information":
				case "Prescription":
				default:
					return PwIcon.Folder;

			}
		}

		private PwUuid ImportId(XAttribute id)
		{
			var bytes = new byte[PwUuid.UuidSize];

			//eWallet IDs are longs. Get the bytes from it and copy them into the Uuid bytes
			BitConverter.GetBytes(((long)id)).CopyTo(bytes, 0);
			//Remaining bytes are left as 0

			return new PwUuid(bytes);
		}
	}
}
