﻿/*
  KeePass Password Safe - The Open-Source Password Manager
  Copyright (C) 2003-2009 Dominik Reichl <dominik.reichl@t-online.de>
  
  CodeWallet3ImportPlugin - Plugin for importing CodeWallet 3 files
  Copyright (C) 2009 Sebastian Baumhekel <sebasbaumh@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using KeePass.Resources;
using KeePassLib;
using KeePassLib.Interfaces;
using KeePassLib.Security;
using KeePassLib.Utility;
using CodeWallet3ImportPlugin.Properties;
using KeePass.DataExchange;

namespace CodeWallet3ImportPlugin
{
	/// <summary>
	/// Import class for code wallet 3 text files.
	/// </summary>
	public class CodeWalletTxt300 : FileFormatProvider
	{
		public override bool SupportsImport
		{
			get
			{
				//only import is supported here
				return true;
			}
		}

		public override bool SupportsExport
		{
			get
			{
				//only import is supported here
				return false;
			}
		}

		public override String FormatName
		{
			get
			{
				//the format name like the newer CodeWallet 6+ name in the KeePass core
				return "CodeWallet TXT 3.00";
			}
		}

		public override String DefaultExtension
		{
			get
			{
				//default extension for Code Wallet export files is txt
				return "txt";
			}
		}

		public override String ApplicationGroup
		{
			get
			{
				//Code Wallet is a password manager
				return KPRes.PasswordManagers;
			}
		}

		public override bool ImportAppendsToRootGroupOnly
		{
			get
			{
				//categories are imported - so this should be false
				return false;
			}
		}

		public override Image SmallIcon
		{
			get
			{
				//the same image as used by the KeePass core
				return Resources.B16x16_Imp_CWallet;
			}
		}
		/// <summary>
		/// Splits a line like "Key: Value" into "Key", "Value".
		/// </summary>
		/// <param name="sLine">Line to split</param>
		/// <param name="sKey">Key</param>
		/// <param name="sValue">Value</param>
		/// <returns>true if at least a key was found, else false</returns>
		private static bool SplitValue(String sLine, out String sKey, out String sValue)
		{
			sKey = String.Empty;
			sValue = String.Empty;
			if (sLine.Length > 0)
			{
				int i = sLine.IndexOf(":");
				if (i != -1)
				{
					sKey = sLine.Substring(0, i);
					//is there data in this line?
					if (i + 3 < sLine.Length)
					{
						sValue = sLine.Substring(i + 2);
					}
					else
					{
						sValue = String.Empty;
					}
					return true;
				}
			}
			return false;
		}

		public override void Import(PwDatabase pwStorage, Stream sInput, IStatusLogger slLogger)
		{
			String[] saLines;
			using (StreamReader sr = new StreamReader(sInput,System.Text.Encoding.Default))
			{
				String strData = sr.ReadToEnd();
				sr.Close();
				saLines = strData.Split(new String[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
			}

			Dictionary<String, PwGroup> dGroupMapping = new Dictionary<String, PwGroup>();
			//add all root level groups to the mapping (if multiple groups have the same name, the last one wins)
			foreach (PwGroup g in pwStorage.RootGroup.Groups)
			{
				dGroupMapping[g.Name] = g;
			}
			
			//root is the group if no category is found
			PwGroup gActiveGroup = pwStorage.RootGroup;
			PwEntry eActiveEntry = null;
			bool bInNotes = false;
			bool bLastLineWasEmpty = false;
			String sKey, sData;

			foreach(String sLine in saLines)
			{
				if (bInNotes)
				{
					if (bLastLineWasEmpty)
					{
						if (SplitValue(sLine, out sKey, out sData))
						{
							//if next is category or card then there is no more notes data
							if ((sKey == "Category") || (sKey == "Card"))
							{
								bInNotes = false;
							}
						}
					}
					//if still in nodes mode
					if (bInNotes)
					{
						if (sLine.Length == 0)
						{
							bLastLineWasEmpty = true;
						}

						//get existing data from notes field
						String sOldData = eActiveEntry.Strings.ReadSafe(PwDefs.NotesField);
						//if it contains data, add a new line
						if (sOldData.Length > 0)
						{
							sOldData += MessageService.NewLine;
						}
						//add actual line
						sOldData += sLine;
						//set new data for notes field
						eActiveEntry.Strings.Set(PwDefs.NotesField, new ProtectedString(
							pwStorage.MemoryProtection.ProtectNotes, sOldData));
					}
				}

				if (!bInNotes)
				{
					if (sLine.Length > 0)
					{
						if (SplitValue(sLine, out sKey, out sData))
						{
							switch (sKey)
							{
								//new category?
								case "Category":
								{
									//split "(# cards)" from end of category title
									int i = sData.LastIndexOf("(");
									if (i != -1)
									{
										sData = sData.Substring(0, i - 1);
									}
									if (!dGroupMapping.TryGetValue(sData, out gActiveGroup))
									{
										//create new group for category and add it
										gActiveGroup = new PwGroup(true, true, sData, PwIcon.Folder);
										pwStorage.RootGroup.AddGroup(gActiveGroup, true);
										//add it to mapping
										dGroupMapping[sData] = gActiveGroup;
									}
									continue;
								} 
								case "Card":
								{
									//reset active entry
									eActiveEntry = null;
									//try to find existing entry (take the first one)
									foreach (PwEntry e in gActiveGroup.Entries)
									{
										//compare title
										if (e.Strings.Get(PwDefs.TitleField).ReadString() == sData)
										{
											eActiveEntry = e;
											break;
										}
									}
									//create new entry if no entry was found
									if (eActiveEntry == null)
									{
										eActiveEntry = new PwEntry(true, true);
										gActiveGroup.AddEntry(eActiveEntry, true);
										eActiveEntry.Strings.Set(PwDefs.TitleField, new ProtectedString(
											pwStorage.MemoryProtection.ProtectTitle, sData));
									}
									continue;
								}
							}
							//check fields that are only valid inside of an entry
							if (eActiveEntry != null)
							{
								switch (sKey)
								{
									case "User ID":
									{
										eActiveEntry.Strings.Set(PwDefs.UserNameField, new ProtectedString(
											pwStorage.MemoryProtection.ProtectUserName, sData));
									}
									break;
									case "Password":
									{
										eActiveEntry.Strings.Set(PwDefs.PasswordField, new ProtectedString(
											pwStorage.MemoryProtection.ProtectPassword, sData));
									}
									break;
									case "Web site":
									{
										eActiveEntry.Strings.Set(PwDefs.UrlField, new ProtectedString(
											pwStorage.MemoryProtection.ProtectUrl, sData));
									}
									break;
									case "Notes":
									{
										eActiveEntry.Strings.Set(PwDefs.NotesField, new ProtectedString(
											pwStorage.MemoryProtection.ProtectNotes, sData));
										bInNotes = true;
										bLastLineWasEmpty = false;
									}
									break;
									default:
									{
										//extra fields go into extra fields in the entry
										eActiveEntry.Strings.Set(sKey, new ProtectedString(
											false, sData));
									}
									break;
								}
							}
						}
					}
				}
			}
		}
	}
}
