﻿/*
  ArcFourCipher - KeePass Plugin
  Copyright (C) 2006-2024 Dominik Reichl <dominik.reichl@t-online.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

using System;
using System.Collections.Generic;
using System.Security.Cryptography;
using System.Text;

using KeePassLib.Utility;

namespace ArcFourCipher
{
	internal sealed class ArcFourTransform : ICryptoTransform
	{
		private readonly byte[] m_s = new byte[256];
		private byte m_i = 0;
		private byte m_j = 0;

		public bool CanReuseTransform { get { return true; } }
		public bool CanTransformMultipleBlocks { get { return true; } }

		// ArcFour is a stream cipher with block size 1; for performance
		// reasons, we declare a larger block size though
		public int InputBlockSize { get { return 128; } }
		public int OutputBlockSize { get { return 128; } }

		public ArcFourTransform(byte[] pbKey, byte[] pbIV)
		{
			if(pbKey == null) throw new ArgumentNullException("pbKey");
			if(pbIV == null) throw new ArgumentNullException("pbIV");

			int cbKey = pbKey.Length, cbIV = pbIV.Length;
			int cbCK = cbKey + cbIV;
			if((cbCK <= 0) || (cbCK > 256)) throw new ArgumentException();

			byte[] pbCK = new byte[cbCK];
			Array.Copy(pbKey, 0, pbCK, 0, cbKey);
			Array.Copy(pbIV, 0, pbCK, cbKey, cbIV);

			for(int i = 0; i < 256; ++i) m_s[i] = (byte)i;

			byte j = 0;
			for(int i = 0; i < 256; ++i)
			{
				j += (byte)(m_s[i] + pbCK[i % cbCK]);

				byte t = m_s[i];
				m_s[i] = m_s[j];
				m_s[j] = t;
			}

			MemUtil.ZeroByteArray(pbCK);

			// In the case of the ArcFour encryption algorithm, discarding the
			// first few bytes increases the security; see cryptanalysis
			byte[] pb = new byte[3072];
			TransformBlock(pb, 0, pb.Length, pb, 0);
		}

		public void Dispose()
		{
			MemUtil.ZeroByteArray(m_s);
			m_i = 0;
			m_j = 0;
		}

		public int TransformBlock(byte[] inputBuffer, int inputOffset,
			int inputCount, byte[] outputBuffer, int outputOffset)
		{
			if(inputBuffer == null) throw new ArgumentNullException("inputBuffer");
			if(inputOffset < 0) throw new ArgumentOutOfRangeException("inputOffset");
			if(inputCount < 0) throw new ArgumentOutOfRangeException("inputCount");
			if(inputCount > (inputBuffer.Length - inputOffset))
				throw new ArgumentException();
			if(outputBuffer == null) throw new ArgumentNullException("outputBuffer");
			if(outputOffset < 0) throw new ArgumentOutOfRangeException("outputOffset");
			if(inputCount > (outputBuffer.Length - outputOffset))
				throw new ArgumentException();

			byte[] s = m_s;
			byte i = m_i, j = m_j;

			for(int d = 0; d < inputCount; ++d)
			{
				++i;
				byte si = s[i];

				j += si;
				byte sj = s[j];

				s[i] = sj;
				s[j] = si;

				byte r = (byte)(si + sj);
				outputBuffer[outputOffset + d] = (byte)(inputBuffer[
					inputOffset + d] ^ s[r]);
			}

			m_i = i;
			m_j = j;
			return inputCount;
		}

		public byte[] TransformFinalBlock(byte[] inputBuffer, int inputOffset,
			int inputCount)
		{
			if(inputCount < 0) throw new ArgumentOutOfRangeException("inputCount");

			byte[] pbOut = new byte[inputCount];
			TransformBlock(inputBuffer, inputOffset, inputCount, pbOut, 0);
			return pbOut;
		}
	}
}
