/*
  Copyright (c) 2008-2009, Dominik Reichl <dominik.reichl@t-online.de>
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  - Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.
*/

#include "RmvDup.h"
#include <tchar.h>
#include <assert.h>
#include <vector>

static IKpAPI* g_pAPI = NULL;

static KP_MENU_ITEM g_menuItems[RD_MENUITEMS_COUNT];

KP_IMPL_STDREFIMPL(CRmvDupPluginImpl)

BOOL WINAPI DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	UNREFERENCED_PARAMETER(hinstDLL);
	UNREFERENCED_PARAMETER(fdwReason);
	UNREFERENCED_PARAMETER(lpvReserved);

	return TRUE;
}

KP_EXPORT HRESULT KP_API KP_I_INITIALIZELIB_DECL(IKpUnknown* pAPI)
{
	UNREFERENCED_PARAMETER(pAPI);
	return S_OK;
}

KP_EXPORT HRESULT KP_API KP_I_RELEASELIB_DECL(IKpUnknown* pAPI)
{
	UNREFERENCED_PARAMETER(pAPI);
	SAFE_RELEASE(g_pAPI);
	return S_OK;
}

KP_EXPORT HRESULT KP_API KP_I_CREATEINSTANCE_DECL(REFIID riid, void** ppvObject,
	IKpUnknown* pAPI)
{
	*ppvObject = NULL;

	if(pAPI == NULL) return E_UNEXPECTED; // Require API

	if((riid == IID_IKpUnknown) || (riid == IID_IKpPlugin))
	{
		if(pAPI->QueryInterface(IID_IKpAPI, (void**)&g_pAPI) != S_OK)
			return E_FAIL;

		IKpPlugin* pImpl = new CRmvDupPluginImpl();
		*ppvObject = ((riid == IID_IKpPlugin) ? pImpl : (IKpUnknown*)pImpl);
		return S_OK;
	}

	return E_NOINTERFACE;
}

CRmvDupPluginImpl::CRmvDupPluginImpl()
{
	KP_IMPL_CONSTRUCT;

	ZeroMemory(&g_menuItems[0], sizeof(KP_MENU_ITEM) * RD_MENUITEMS_COUNT);

	g_menuItems[RD_MENUITEMID_EXEC].lpCommandString = _T("Remove Duplicate Entries");
	g_menuItems[RD_MENUITEMID_EXEC].dwIcon = 44;

	g_menuItems[RD_MENUITEMID_RMVEMPTYGROUPS].lpCommandString = _T("Remove Empty Groups");
	g_menuItems[RD_MENUITEMID_RMVEMPTYGROUPS].dwIcon = 44;
}

STDMETHODIMP CRmvDupPluginImpl::QueryInterface(REFIID riid, void** ppvObject)
{
	KP_SUPPORT_INTERFACE(IID_IKpUnknown, IKpUnknown);
	KP_SUPPORT_INTERFACE(IID_IKpPlugin, IKpPlugin);

	*ppvObject = NULL;
	return E_NOINTERFACE;
}

STDMETHODIMP_(BOOL) CRmvDupPluginImpl::OnMessage(DWORD dwCode, LPARAM lParamW,
	LPARAM lParamL)
{
	UNREFERENCED_PARAMETER(lParamL);

	switch(dwCode)
	{
	case KPM_DIRECT_CONFIG:
		MessageBox(NULL, _T("This plugin doesn't have any options."),
			RD_PLUGIN_NAME, MB_OK | MB_ICONINFORMATION);
		break;

	case KPM_PLUGIN_INFO:
		MessageBox(NULL, _T("This plugin can remove all duplicate entries and empty groups."),
			RD_PLUGIN_NAME, MB_OK | MB_ICONINFORMATION);
		break;

	case KPM_DIRECT_EXEC:
		if(lParamW == (LPARAM)g_menuItems[RD_MENUITEMID_EXEC].dwCommandID)
			RemoveDuplicateEntriesEx();
		else if(lParamW == (LPARAM)g_menuItems[RD_MENUITEMID_RMVEMPTYGROUPS].dwCommandID)
			RemoveEmptyGroupsEx();
		break;

	case KPM_INIT_MENU_POPUP:
		if((g_pAPI != NULL) && (g_pAPI->IsFileOpen() == TRUE))
		{
			g_menuItems[RD_MENUITEMID_EXEC].dwFlags &= ~KPMIF_DISABLED;
			g_menuItems[RD_MENUITEMID_RMVEMPTYGROUPS].dwFlags &= ~KPMIF_DISABLED;
		}
		else
		{
			g_menuItems[RD_MENUITEMID_EXEC].dwFlags |= KPMIF_DISABLED;
			g_menuItems[RD_MENUITEMID_RMVEMPTYGROUPS].dwFlags |= KPMIF_DISABLED;
		}
		break;

	default:
		break;
	}

	return TRUE;
}

STDMETHODIMP_(LPCTSTR) CRmvDupPluginImpl::GetProperty(LPCTSTR lpName)
{
	UNREFERENCED_PARAMETER(lpName);
	return NULL;
}

STDMETHODIMP CRmvDupPluginImpl::SetProperty(LPCTSTR lpName, LPCTSTR lpValue)
{
	UNREFERENCED_PARAMETER(lpName);
	UNREFERENCED_PARAMETER(lpValue);
	return E_FAIL;
}

STDMETHODIMP_(DWORD) CRmvDupPluginImpl::GetMenuItemCount()
{
	return RD_MENUITEMS_COUNT;
}

STDMETHODIMP_(KP_MENU_ITEM*) CRmvDupPluginImpl::GetMenuItems()
{
	return &g_menuItems[0];
}

void RemoveDuplicateEntriesEx()
{
	if(g_pAPI->IsFileOpen() == FALSE) return;

	g_pAPI->SetDisplayDialog(TRUE);

	IKpDatabase* pDB = NULL;
	if(g_pAPI->QueryInstance(SCLSID_KpDatabase, IID_IKpDatabase, (void**)&pDB) != S_OK)
		return;

	DWORD dwIndex = 0, dwRemoved = 0;
	while(dwIndex < pDB->GetEntryCount())
	{
		PW_ENTRY* peCurrent = pDB->GetEntry(dwIndex);

		if(IsBackupPwEntry(pDB, peCurrent) == TRUE)
		{
			++dwIndex;
			continue;
		}

		std::vector<DWORD> vRemove;
		for(DWORD i = (dwIndex + 1); i < pDB->GetEntryCount(); ++i)
		{
			PW_ENTRY* peCompare = pDB->GetEntry(i);

			if(IsBackupPwEntry(pDB, peCompare) == TRUE)
				continue;

			if(PwEntryEquals(pDB, peCurrent, peCompare) == TRUE)
				vRemove.push_back(i);
		}

		if(vRemove.size() > 0)
		{
			for(size_t iRemove = 0; iRemove < vRemove.size(); ++iRemove)
			{
				DWORD dwEntryIndex = vRemove[vRemove.size() - iRemove - 1];
				
				pDB->DeleteEntry(dwEntryIndex);

				++dwRemoved;
			}

			vRemove.clear();
		}

		++dwIndex;
	}

	g_pAPI->SetDisplayDialog(FALSE);

	if(dwRemoved > 0)
	{
		g_pAPI->SetFileModified(TRUE);
		g_pAPI->UpdateUI();
		g_pAPI->UpdateToolBar();

		TCHAR tszNumber[34];
		_ltot_s(static_cast<long>(dwRemoved), tszNumber, 32, 10);

		std::basic_string<TCHAR> strMsg;
		strMsg += _T("Successfully removed ");
		strMsg += tszNumber;
		strMsg += _T(" duplicate entries!");

		MessageBox(g_pAPI->GetMainWindowHandle(), strMsg.c_str(),
			RD_PLUGIN_NAME, MB_ICONINFORMATION | MB_OK);
	}
	else
		MessageBox(g_pAPI->GetMainWindowHandle(), _T("No duplicate entries found!"),
			RD_PLUGIN_NAME, MB_ICONINFORMATION | MB_OK);

	pDB->Release();
}

BOOL PwEntryEquals(IKpDatabase* pDB, PW_ENTRY* pe1, PW_ENTRY* pe2)
{
	if((pe1 == NULL) || (pe2 == NULL)) return FALSE;

	if(_tcscmp(pe1->pszTitle, pe2->pszTitle) != 0) return FALSE;
	if(_tcscmp(pe1->pszURL, pe2->pszURL) != 0) return FALSE;
	if(_tcscmp(pe1->pszUserName, pe2->pszUserName) != 0) return FALSE;
	if(_tcscmp(pe1->pszAdditional, pe2->pszAdditional) != 0) return FALSE;

	pDB->UnlockEntryPassword(pe1);
	pDB->UnlockEntryPassword(pe2);
	if(_tcscmp(pe1->pszPassword, pe2->pszPassword) != 0)
	{
		pDB->LockEntryPassword(pe2);
		pDB->LockEntryPassword(pe1);

		return FALSE;
	}
	pDB->LockEntryPassword(pe2);
	pDB->LockEntryPassword(pe1);

	return TRUE;
}

BOOL IsBackupPwEntry(IKpDatabase* pDB, PW_ENTRY* pe)
{
	if(pe == NULL) return FALSE;

	PW_GROUP* pgCurrent = pDB->GetGroupById(pe->uGroupId);
	if(pgCurrent == NULL) return FALSE;

	if((_tcscmp(pgCurrent->pszGroupName, pDB->GetBackupGroupName(FALSE)) == 0) ||
		(_tcscmp(pgCurrent->pszGroupName, pDB->GetBackupGroupName(TRUE)) == 0))
	{
		return TRUE;
	}

	return FALSE;
}

void RemoveEmptyGroupsEx()
{
	if(g_pAPI->IsFileOpen() == FALSE) return;

	IKpDatabase* pDB = NULL;
	if(g_pAPI->QueryInstance(SCLSID_KpDatabase, IID_IKpDatabase, (void**)&pDB) != S_OK)
		return;

	g_pAPI->SetDisplayDialog(TRUE);

	DWORD dwRemoved = 0;
	for(long i = static_cast<long>(pDB->GetGroupCount()) - 1; i >= 0; --i)
	{
		const DWORD dwIndex = static_cast<DWORD>(i);
		const DWORD dwId = pDB->GetGroupIdByIndex(dwIndex);
		const DWORD dwEntries = pDB->GetEntryCountInGroupN(dwId);

		if(dwEntries == 0)
		{
			pDB->DeleteGroupById(dwId, FALSE);
			++dwRemoved;
		}
	}

	g_pAPI->SetDisplayDialog(FALSE);

	if(dwRemoved > 0)
	{
		g_pAPI->SetFileModified(TRUE);
		g_pAPI->UpdateUI();
		g_pAPI->UpdateToolBar();

		TCHAR tszNumber[34];
		_ltot_s(static_cast<long>(dwRemoved), tszNumber, 32, 10);

		std::basic_string<TCHAR> strMsg;
		strMsg += _T("Successfully removed ");
		strMsg += tszNumber;
		strMsg += _T(" empty groups!");

		MessageBox(g_pAPI->GetMainWindowHandle(), strMsg.c_str(),
			RD_PLUGIN_NAME, MB_ICONINFORMATION | MB_OK);
	}
	else
		MessageBox(g_pAPI->GetMainWindowHandle(), _T("No empty groups found!"),
			RD_PLUGIN_NAME, MB_ICONINFORMATION | MB_OK);

	pDB->Release();
}
