/*
  KeePass On-Screen Keyboard Plugin
  Copyright (C) 2005-2009 Dominik Reichl <dominik.reichl@t-online.de>
                     2011 Nikolaus Hammler <nikolaus@hammler.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "StdAfx.h"
#include "OSKPlugin.h"
#include "SettingsDlg.h"
#include "InfoDlg.h"
#include "KeyboardWindows.h"
#include "KeyboardBuiltin.h"
#include <assert.h>

IKpAPI* g_pAPI = NULL;

KP_IMPL_STDREFIMPL(COSKPluginImpl)

KP_EXPORT HRESULT KP_API KP_I_INITIALIZELIB_DECL(IKpUnknown* pAPI)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	UNREFERENCED_PARAMETER(pAPI);
	return S_OK;
}

KP_EXPORT HRESULT KP_API KP_I_RELEASELIB_DECL(IKpUnknown* pAPI)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	UNREFERENCED_PARAMETER(pAPI);
	SAFE_RELEASE(g_pAPI);
	return S_OK;
}

KP_EXPORT HRESULT KP_API KP_I_CREATEINSTANCE_DECL(REFIID riid, void** ppvObject,
	IKpUnknown* pAPI)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	*ppvObject = NULL;

	if(pAPI == NULL) return E_UNEXPECTED; // Require API

	if((riid == IID_IKpUnknown) || (riid == IID_IKpPlugin))
	{
		if(pAPI->QueryInterface(IID_IKpAPI, (void**)&g_pAPI) != S_OK)
			return E_FAIL;

		IKpPlugin* pImpl = new COSKPluginImpl();
		*ppvObject = ((riid == IID_IKpPlugin) ? pImpl : (IKpUnknown*)pImpl);
		return S_OK;
	}

	return E_NOINTERFACE;
}

COSKPluginImpl::COSKPluginImpl()
	: m_keyboard(0)
	, m_iUseKeyboard(0)
	, m_bDetectTPC(FALSE)
{
	KP_IMPL_CONSTRUCT;

	LoadSettings();
	if(m_bDetectTPC)
		m_objStylusWatcher.InstallHook();

	ZeroMemory(&m_menuItems[0], sizeof(KP_MENU_ITEM) * OSK_MENUITEMCOUNT);

	m_menuItems[0].lpCommandString = _T("On-Screen Keyboard");
	m_menuItems[0].dwFlags = KPMIF_POPUP_START;

	m_menuItems[OSK_MENUITEMID_EXEC].lpCommandString = _T("&Show On-Screen Keyboard");
	m_menuItems[OSK_MENUITEMID_EXEC].dwIcon = 34;
	m_menuItems[OSK_MENUITEMID_EXEC + 1].lpCommandString = _T(""); // Separator

	m_menuItems[OSK_MENUITEMID_SETTINGS].lpCommandString = _T("Se&ttings");

	m_menuItems[OSK_MENUITEMID_HELP].lpCommandString = _T("&Help");
	m_menuItems[OSK_MENUITEMID_HELP].dwIcon = 8;

	m_menuItems[OSK_MENUITEMID_HELP + 1].lpCommandString = _T("");
	m_menuItems[OSK_MENUITEMID_HELP + 1].dwFlags = KPMIF_POPUP_END;

	ReCreateKeyboard();
}

COSKPluginImpl::~COSKPluginImpl()
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	m_objStylusWatcher.UninstallHook();

	SaveSettings();

	delete m_keyboard;
	m_keyboard = 0;
}

STDMETHODIMP COSKPluginImpl::QueryInterface(REFIID riid, void** ppvObject)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	KP_SUPPORT_INTERFACE(IID_IKpUnknown, IKpUnknown);
	KP_SUPPORT_INTERFACE(IID_IKpPlugin, IKpPlugin);

	*ppvObject = NULL;
	return E_NOINTERFACE;
}

void COSKPluginImpl::LoadSettings()
{
	IKpConfig* p = NULL;
	if(g_pAPI->CreateInstance(CLSID_KpConfig_ReadOnly, IID_IKpConfig, (void**)&p) == S_OK)
	{
		char str_cfg[SI_REGSIZE];
		if(p->Get(_T("OSK_UseKeyboard"), str_cfg))
			m_iUseKeyboard = atol(str_cfg);

		m_bDetectTPC = p->GetBool(_T("OSK_DetectTPC"), FALSE);
		VERIFY(p->Release() == 0);
	}
}

void COSKPluginImpl::SaveSettings()
{
	IKpConfig* p = NULL;
	if(g_pAPI->CreateInstance(CLSID_KpConfig, IID_IKpConfig, (void**)&p) == S_OK)
	{
		char str_cfg[SI_REGSIZE];
		// FIXXME: do it better
		wsprintf(str_cfg, _T("%d"), m_iUseKeyboard);
		p->Set(_T("OSK_UseKeyboard"), str_cfg);

		p->SetBool(_T("OSK_DetectTPC"), m_bDetectTPC);
		VERIFY(p->Release() == 0);
	}
}

void COSKPluginImpl::ReCreateKeyboard()
{
	BOOL bUseBuiltinOSK;
	OSVERSIONINFO os;
	ZeroMemory(&os, sizeof(OSVERSIONINFO));
	os.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	VERIFY(GetVersionEx(&os) != FALSE);
	bUseBuiltinOSK = ((os.dwMajorVersion >= 5) ? FALSE : TRUE);

	if(m_keyboard) {
		delete m_keyboard; m_keyboard = 0;
	}

	switch(m_iUseKeyboard)
	{
		case 0:
			if(bUseBuiltinOSK)
				m_keyboard = new CKeyboardBuiltin();
			else
				m_keyboard = new CKeyboardWindows();
			break;
		case 1:
			m_keyboard = new CKeyboardWindows();
			break;
		case 2:
			m_keyboard = new CKeyboardBuiltin();
			break;
	}
}

void COSKPluginImpl::ShowSettings()
{
	SettingsDlg dlg;
	dlg.m_bDetectTPC = m_bDetectTPC;
	dlg.m_iKeyboard = m_iUseKeyboard;

	if(dlg.DoModal() != IDOK)
		return;

	m_bDetectTPC = dlg.m_bDetectTPC;
	m_iUseKeyboard = dlg.m_iKeyboard;
	ReCreateKeyboard();
	SaveSettings();
	if(m_bDetectTPC)
		m_objStylusWatcher.InstallHook();
	else
		m_objStylusWatcher.UninstallHook();
}

STDMETHODIMP_(BOOL) COSKPluginImpl::OnMessage(DWORD dwCode, LPARAM lParamW, LPARAM lParamL)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	switch(dwCode)
	{
		case KPM_DIRECT_CONFIG:
			ShowSettings();
			break;
		case KPM_PLUGIN_INFO:
			{
				CInfoDlg dlg;
				dlg.DoModal();
			}
			break;
		case KPM_OPENDB_PRE:
			if(m_bDetectTPC && !m_objStylusWatcher.IsTabletMode())
				break;
			m_keyboard->ShowKeyboard();
			break;

		case KPM_OPENDB_POST:
			m_keyboard->HideKeyboard();
			break;

		case KPM_DIRECT_EXEC:
			My_ProcessDirectCall(lParamW, lParamL);
			break;
	}
	return TRUE;
}

STDMETHODIMP_(LPCTSTR) COSKPluginImpl::GetProperty(LPCTSTR lpName)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	UNREFERENCED_PARAMETER(lpName);
	return NULL;
}

STDMETHODIMP COSKPluginImpl::SetProperty(LPCTSTR lpName, LPCTSTR lpValue)
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());

	UNREFERENCED_PARAMETER(lpName);
	UNREFERENCED_PARAMETER(lpValue);
	return E_FAIL;
}

STDMETHODIMP_(DWORD) COSKPluginImpl::GetMenuItemCount()
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());
	return OSK_MENUITEMCOUNT;
}

STDMETHODIMP_(KP_MENU_ITEM*) COSKPluginImpl::GetMenuItems()
{
	AFX_MANAGE_STATE(AfxGetStaticModuleState());
	return &m_menuItems[0];
}

void COSKPluginImpl::My_ProcessDirectCall(LPARAM lParamW, LPARAM lParamL)
{
	UNREFERENCED_PARAMETER(lParamL);

	if(lParamW == (LPARAM)m_menuItems[OSK_MENUITEMID_EXEC].dwCommandID) {
		m_keyboard->OpenKeyboard();
	}
	else if(lParamW == (LPARAM)m_menuItems[OSK_MENUITEMID_SETTINGS].dwCommandID) {
		ShowSettings();
	}
	else if(lParamW == (LPARAM)m_menuItems[OSK_MENUITEMID_HELP].dwCommandID) {
		IKpUtilities* pUtilities = NULL;
		if(g_pAPI->QueryInstance(SCLSID_KpUtilities, IID_IKpUtilities, (void**)&pUtilities) == S_OK)
		{
			pUtilities->ShellOpenLocalFile(_T("OnScreenKeyboard_ReadMe.txt"), OLF_OPEN);
			pUtilities->Release();
		}
	}
}

