/*
  KeePass On-Screen Keyboard Plugin
  Copyright (C) 2005-2009 Dominik Reichl <dominik.reichl@t-online.de>
                     2011 Nikolaus Hammler <nikolaus@hammler.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

// This on-screen keyboard for KeePass is based on Randy More's
// on-screen keyboard implementation for pen computing and touchscreens.
// You can find his original sources and article here:
// http://www.codeproject.com/samples/onscreenkeyboard.asp

#include "StdAfx.h"
#include "OnScreenKeyboard.h"
#include "OSKDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define OSK_TIMER_ID 5

#define OSK_TAB 0x02
#define OSK_CAPSLOCK 0x03
#define OSK_SHIFT 0x04
#define OSK_CTRL 0x05
#define OSK_BSPACE 0x06
#define OSK_ENTER 0x07
#define OSK_ALT 0x08
#define OSK_WINDOW 0x9
#define OSK_LEFT 0x0A
#define OSK_RIGHT 0x0B
#define OSK_UP 0x0C
#define OSK_DOWN 0x0D
#define OSK_ESC 0x0E
#define OSK_PUP 0x0F
#define OSK_PDN 0x10
#define OSK_HOME 0x11
#define OSK_END 0x12
#define OSK_INS 0x13
#define OSK_DEL 0x14
#define OSK_WWW 0x15

/////////////////////////////////////////////////////////////////////////////

COSKDlg::COSKDlg(CWnd* pParent /*=NULL*/)
	: CDialog(COSKDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(COSKDlg)
	//}}AFX_DATA_INIT
}

void COSKDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(COSKDlg)
	//}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(COSKDlg, CDialog)
	//{{AFX_MSG_MAP(COSKDlg)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONDBLCLK()
	ON_WM_RBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_MOUSEWHEEL()
	ON_WM_LBUTTONUP()
	ON_WM_SIZE()
	ON_WM_SHOWWINDOW()
	ON_WM_TIMER()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////

BOOL COSKDlg::OnInitDialog() 
{
	CDialog::OnInitDialog();

	AddKey(0x00, 0x00, 0);
	AddKey(0x00, OSK_ESC, 10);

	int fnKey;
	for(fnKey = 0x70; fnKey < 0x7C; fnKey++)
		AddKey(0x00, (char)(fnKey & 0xFF), 10);

	AddKey(0x00, OSK_INS, 14);
	AddKey(0x00, OSK_DEL, -10);

	AddKey(0x00, 0x00, 1); // Signal row change
	AddKey('`', '~', 10);
	AddKey('1', '!', 10);
	AddKey('2', '@', 10);
	AddKey('3', '#', 10);
	AddKey('4', '$', 10);
	AddKey('5', '%', 10);
	AddKey('6', '^', 10);
	AddKey('7', '&', 10);
	AddKey('8', '*', 10);
	AddKey('9', '(', 10);
	AddKey('0', ')', 10);
	AddKey('-', '_', 10);
	AddKey('=', '+', 10);
	AddKey(0x00, OSK_BSPACE, 14);
	AddKey(0x00, OSK_PUP, -12);

	AddKey(0x00, 0x00, 2); // Signal row change
	AddKey(0x00, OSK_TAB, 14);
	AddKey('q', 'Q', 10);
	AddKey('w', 'W', 10);
	AddKey('e', 'E', 10);
	AddKey('r', 'R', 10);
	AddKey('t', 'T', 10);
	AddKey('y', 'Y', 10);
	AddKey('u', 'U', 10);
	AddKey('i', 'I', 10);
	AddKey('o', 'O', 10);
	AddKey('p', 'P', 10);
	AddKey('[', '{', 10);
	AddKey(']', '}', 10);
	AddKey('\\', '|', 10);
	AddKey(0x00, OSK_PDN, -12);

	AddKey(0x00, 0x00, 3); // Signal row change
	AddKey(0x00, OSK_CAPSLOCK, 18);
	AddKey('a', 'A', 10);
	AddKey('s', 'S', 10);
	AddKey('d', 'D', 10);
	AddKey('f', 'F', 10);
	AddKey('g', 'G', 10);
	AddKey('h', 'H', 10);
	AddKey('j', 'J', 10);
	AddKey('k', 'K', 10);
	AddKey('l', 'L', 10);
	AddKey(';', ':', 10);
	AddKey('\'', '\"', 10);
	AddKey(0x00, OSK_ENTER, 16);
	AddKey(0x00, OSK_HOME, -10);

	AddKey(0x00, 0x00, 4); // Signal row change
	AddKey(0x00, OSK_SHIFT, 22);
	AddKey('z', 'Z', 10);
	AddKey('x', 'X', 10);
	AddKey('c', 'C', 10);
	AddKey('v', 'V', 10);
	AddKey('b', 'B', 10);
	AddKey('n', 'N', 10);
	AddKey('m', 'M', 10);
	AddKey(',', '<', 10);
	AddKey('.', '>', 10);
	AddKey('/', '?', 10);
	AddKey(0x00, OSK_SHIFT, 22);
	AddKey(0x00, OSK_END, -10);

	AddKey(0x00, 0x00, 5); // Signal row change
	AddKey(0x00, OSK_CTRL, 12);
	AddKey(0x00, OSK_WINDOW, 12);
	AddKey(0x00, OSK_ALT, 12);
	AddKey(' ', ' ', 60);
	AddKey(0x00, OSK_LEFT, 12);
	AddKey(0x00, OSK_UP, 12);
	AddKey(0x00, OSK_DOWN, 12);
	AddKey(0x00, OSK_RIGHT, 12);
	AddKey(0x00, OSK_WWW, -10);

	CalcWidthHeight();

	SetTimer(OSK_TIMER_ID, 250, NULL);

	return TRUE;
}

void COSKDlg::CleanUp()
{
	KillTimer(OSK_TIMER_ID);

	POSITION pos = cKeys.GetHeadPosition();
	while(pos)
	{
		KEYDEF *pKey = cKeys.GetNext(pos);
		ASSERT(pKey != NULL); if(pKey == NULL) continue;

		delete pKey; pKey = NULL;
	}

	cKeys.RemoveAll();
}

void COSKDlg::OnOK() 
{
	// CDialog::OnOK();
	ShowWindow(SW_HIDE);
}

void COSKDlg::OnCancel() 
{
	// CDialog::OnCancel();
	ShowWindow(SW_HIDE);
}

void COSKDlg::PumpMessages()
{
	MSG msg;

	while(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
}

void COSKDlg::CalcWidthHeight()
{
	KEYDEF *pKey;

	nKeyWidth = 0;
	nKeyHeight = 0;

	POSITION pos = cKeys.GetHeadPosition();
	long totalWidth = 0;
	long nLongest = 0;
	long nRows = 0;

	while(pos)
	{
		pKey = cKeys.GetNext(pos);
		ASSERT(pKey != NULL); if(pKey == NULL) continue;

		if((pKey->cNormal == 0x00) && (pKey->cShifted == 0x00))
		{
			nRows++;

			if(totalWidth > nLongest) nLongest = totalWidth;
			totalWidth = 0;
		}
		else totalWidth += abs(pKey->cWidth);
	}

	ASSERT(nRows != 0); if(nRows == 0) nRows = 1;
	ASSERT(nLongest != 0); if(nLongest == 0) nLongest = 10;

	CRect rect;
	GetClientRect(rect);
	nKeyHeight = rect.Height() / nRows;
	nKeyWidth = (int)(((double)rect.Width()) / ((((double)(nLongest)) / 10)));
}

void COSKDlg::AddKey(char chNormal, char chShifted, int nWidth)
{
	KEYDEF *pKey = new KEYDEF;

	pKey->cNormal = chNormal;
	pKey->cShifted = chShifted;
	pKey->cWidth = nWidth;

	cKeys.AddTail(pKey);
}

void COSKDlg::DrawKey(CDC *pDC, CRect& rc, KEYDEF *pKey, BOOL cHilight)
{
	ASSERT(pDC != NULL); if(pDC == NULL) return;
	ASSERT(pKey != NULL); if(pKey == NULL) return;

	if(cHilight == FALSE)
	{
		rc.DeflateRect(1, 1);
		pDC->FillSolidRect(rc, GetSysColor(COLOR_3DFACE));
		pDC->Draw3dRect(rc, GetSysColor(COLOR_3DSHADOW), GetSysColor(COLOR_3DDKSHADOW));
		rc.DeflateRect(1, 1);
		pDC->Draw3dRect(rc, GetSysColor(COLOR_3DLIGHT), GetSysColor(COLOR_3DSHADOW));
	}
	else
	{
		rc.DeflateRect(1, 1);
		pDC->FillSolidRect(rc, GetSysColor(COLOR_3DLIGHT));
		pDC->Draw3dRect(rc, GetSysColor(COLOR_3DLIGHT), GetSysColor(COLOR_3DSHADOW));
		rc.DeflateRect(1, 1);
		pDC->Draw3dRect(rc, GetSysColor(COLOR_3DSHADOW), GetSysColor(COLOR_3DDKSHADOW));
	}

	rc.DeflateRect(3, 1);

	if(pKey->cNormal == 0x00)
	{
		CString strLabel = " ";
		BOOL bSpecial = FALSE;

		if(pKey->cShifted > 0x6F) // Is a function key?
		{
			strLabel.Format(_T("F%d"), (int)(pKey->cShifted - 0x6F));
		}
		else
		{
			switch(pKey->cShifted)
			{
			case OSK_TAB:
				strLabel = "Tab->";
				break;
			case OSK_CAPSLOCK:
				strLabel = "Caps";
				break;
			case OSK_SHIFT:
				strLabel = "Shift";
				break;
			case OSK_CTRL:
				strLabel = "Ctrl";
				break;
			case OSK_BSPACE:
				strLabel = "<-Bksp";
				break;
			case OSK_ENTER:
				strLabel = "Enter";
				break;
			case OSK_ALT:
				strLabel = "Alt";
				break;
			case OSK_WINDOW:
				strLabel = "Win";
				break;
			case OSK_ESC:
				strLabel = "Esc";
				break;
			case OSK_LEFT:
				bSpecial = TRUE;
				{
					int cx = (rc.left + rc.right) / 2;
					int cy = (rc.top + rc.bottom) / 2;
					int siz = rc.Width() / 3;
					pDC->MoveTo(cx + siz, cy);
					pDC->LineTo(cx - siz, cy);
					pDC->LineTo(cx - siz + ((siz * 4) / 5), cy - (siz / 2));
					pDC->MoveTo(cx - siz, cy);
					pDC->LineTo(cx - siz + ((siz * 4) / 5), cy + (siz / 2));
				}
				break;
			case OSK_RIGHT:
				bSpecial = TRUE;
				{
					int cx = (rc.left + rc.right) / 2;
					int cy = (rc.top + rc.bottom) / 2;
					int siz = rc.Width() / 3;
					pDC->MoveTo(cx - siz, cy);
					pDC->LineTo(cx + siz, cy);
					pDC->LineTo(cx + siz - ((siz * 4) / 5), cy - (siz / 2));
					pDC->MoveTo(cx + siz, cy);
					pDC->LineTo(cx + siz - ((siz * 4) / 5), cy + (siz / 2));
				}
				break;
			case OSK_UP:
				bSpecial = TRUE;
				{
					int cx = (rc.left + rc.right) / 2;
					int cy = (rc.top + rc.bottom) / 2;
					int siz = rc.Width() / 3;
					pDC->MoveTo(cx, cy + siz);
					pDC->LineTo(cx, cy - siz);
					pDC->LineTo(cx - (siz / 2), cy - siz + ((siz * 4) / 5));
					pDC->LineTo(cx, cy - siz);
					pDC->LineTo(cx + (siz / 2), cy - siz + ((siz * 4) / 5));
				}
				break;
			case OSK_DOWN:
				bSpecial = TRUE;
				{
					int cx = (rc.left + rc.right) / 2;
					int cy = (rc.top + rc.bottom) / 2;
					int siz = rc.Width() / 3;
					pDC->MoveTo(cx, cy - siz);
					pDC->LineTo(cx, cy + siz);
					pDC->LineTo(cx - (siz / 2), cy + siz - ((siz * 4) / 5));
					pDC->LineTo(cx, cy + siz);
					pDC->LineTo(cx + (siz / 2), cy + siz - ((siz * 4) / 5));
				}
				break;
			case OSK_PUP:
				strLabel = "PgUp";
				break;
			case OSK_PDN:
				strLabel = "PgDn";
				break;
			case OSK_HOME:
				strLabel = "Home";
				break;
			case OSK_END:
				strLabel = "End";
				break;
			case OSK_INS:
				strLabel = "Ins";
				break;
			case OSK_DEL:
				strLabel = "Del";
				break;
			case OSK_WWW:
				strLabel = "www";
				break;
			default:
				{ ASSERT(FALSE); }
				strLabel = "#ERR#";
				break;
			}
		}
		if(bSpecial == FALSE)
		{
			CSize tsize;
			tsize = pDC->GetTextExtent(strLabel);
			pDC->TextOut(((rc.left + rc.right) / 2) - (tsize.cx / 2),
				((rc.top + rc.bottom) / 2) - (tsize.cy / 2), strLabel);
		}
	}
	else
	{
		if((pKey->cShifted >= 'A') && (pKey->cShifted <= 'Z'))
		{
			CSize tsize;
			tsize = pDC->GetTextExtent(CString(pKey->cShifted));
			pDC->TextOut(((rc.left + rc.right) / 2) - (tsize.cx / 2), 
				((rc.top + rc.bottom) / 2) - (tsize.cy / 2), CString(pKey->cShifted));
		}
		else
		{
			CSize tsize;
			tsize = pDC->GetTextExtent(CString('M'));
			pDC->TextOut(rc.left, rc.top, CString(pKey->cShifted));
			pDC->TextOut(rc.right - tsize.cx, rc.bottom - tsize.cy, CString(pKey->cNormal));
		}
	}
}

void COSKDlg::OnPaint() 
{
	CDialog::OnPaint();
	DrawKeyboard();
}

void COSKDlg::SendKey(KEYDEF *pKeyDef)
{
	ASSERT(pKeyDef != NULL); if(pKeyDef == NULL) return;

	if(pKeyDef->cNormal == 0x00)
	{
		int vk = 0;
		BOOL bUpToo = TRUE;

		if(pKeyDef->cShifted > 0x6F) // Is a function key?
		{
			vk = pKeyDef->cShifted;
		}
		else
		{
			switch(pKeyDef->cShifted)
			{
			case OSK_TAB:
				vk = VkKeyScan(0x09) & 0xFF;
				break;
			case OSK_CAPSLOCK:
				vk = VK_CAPITAL;
				break;
			case OSK_SHIFT:
				vk = VK_SHIFT;
				if(GetKeyState(vk) & 0xF000) return;
				bUpToo = FALSE;
				break;
			case OSK_CTRL:
				vk = VK_CONTROL;
				if(GetKeyState(vk) & 0xF000) return;
				bUpToo = FALSE;
				break;
			case OSK_BSPACE:
				vk = VK_BACK;
				break;
			case OSK_ENTER:
				vk = VkKeyScan(0x0D) & 0xFF;
				break;
			case OSK_ALT:
				vk = VK_MENU;
				break;
			case OSK_WINDOW:
				vk = VK_LWIN;
				break;
			case OSK_LEFT:
				vk = VK_LEFT;
				break;
			case OSK_RIGHT:
				vk = VK_RIGHT;
				break;
			case OSK_UP:
				vk = VK_UP;
				break;
			case OSK_DOWN:
				vk = VK_DOWN;
				break;
			case OSK_ESC:
				vk = VK_ESCAPE;
				break;
			case OSK_PUP:
				vk = VK_PRIOR;
				break;
			case OSK_PDN:
				vk = VK_NEXT;
				break;
			case OSK_HOME:
				vk = VK_HOME;
				break;
			case OSK_END:
				vk = VK_END;
				break;
			case OSK_INS:
				vk = VK_INSERT;
				break;
			case OSK_DEL:
				vk = VK_DELETE;
				break;
			case OSK_WWW:
				KEYDEF keyDef;
				keyDef.cNormal = keyDef.cShifted = 'W';
				SendKey(&keyDef);
				SendKey(&keyDef);
				SendKey(&keyDef);
				vk = VkKeyScan('.') & 0xFF;
				break;
			default:
				ASSERT(FALSE);
				break;
			}
		}

		keybd_event((unsigned char)vk, 0, 0, 0);
		if(bUpToo == TRUE) keybd_event((unsigned char)vk, 0, KEYEVENTF_KEYUP, 0);
	}
	else
	{
		char pChar = pKeyDef->cNormal;
		SHORT ks = VkKeyScan(pChar);
		BYTE btKey = (BYTE)(ks & 0xFF);

		keybd_event(btKey,0,0,0);
		keybd_event(btKey,0,KEYEVENTF_KEYUP,0);

		// Turn off the control and shift if they were down
		unsigned char vk = VK_SHIFT;
		if(GetKeyState(vk) & 0xF000)
		{
			keybd_event(vk,0,KEYEVENTF_KEYUP,0);
		}
		vk = VK_CONTROL;
		if(GetKeyState(vk) & 0xF000) // Allow toggle
		{
			keybd_event(vk,0,KEYEVENTF_KEYUP,0);
		}
	}
}

void COSKDlg::OnLButtonDown(UINT nFlags, CPoint point) 
{
	UNREFERENCED_PARAMETER(nFlags);

	ReleaseFocus();

	CRect client;
	GetClientRect(client);
	int x = 0;
	int y = -nKeyHeight;
	KEYDEF * pKey;
	POSITION pos = cKeys.GetHeadPosition();
	while(pos)
	{
		pKey = cKeys.GetNext(pos);
		if((pKey->cNormal == 0x00) && (pKey->cShifted == 0x00))
		{
			x = 0;
			y += nKeyHeight;
		}
		else
		{
			int width = ((abs(pKey->cWidth) * nKeyWidth) / 10);
			CRect rc(x,y,x + width, y + nKeyHeight);
			if(pKey->cWidth < 0)
			{
				rc.right = client.right;
			}
			x += width;
			if(rc.PtInRect(point))
			{
				SendKey(pKey);
			}
		}
	}
}

void COSKDlg::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
	OnLButtonDown(nFlags, point);
}

BOOL COSKDlg::PreTranslateMessage(MSG* pMsg) 
{
	HWND wnd = ::GetForegroundWindow();
	if(IsWindow(wnd))
	{
		if(wnd != this->m_hWnd)
		{
			if(gFocus != wnd)
			{
				if(IsWindow(gFocus))
				{
					AttachThreadInput(
						GetWindowThreadProcessId(m_hWnd,NULL),
						GetWindowThreadProcessId(gFocus,NULL),
						FALSE);
				}

				gFocus = wnd;
				AttachThreadInput(
					GetWindowThreadProcessId(m_hWnd,NULL),
					GetWindowThreadProcessId(gFocus,NULL),
					TRUE);
			}
		}
	}
	return CDialog::PreTranslateMessage(pMsg);
}

void COSKDlg::OnRButtonDblClk(UINT nFlags, CPoint point) 
{
	OnLButtonDown(nFlags, point);
}

void COSKDlg::OnRButtonDown(UINT nFlags, CPoint point) 
{
	OnLButtonDown(nFlags, point);
}

void COSKDlg::OnRButtonUp(UINT nFlags, CPoint point) 
{
	UNREFERENCED_PARAMETER(nFlags);
	UNREFERENCED_PARAMETER(point);

	ReleaseFocus();
}

BOOL COSKDlg::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
	UNREFERENCED_PARAMETER(nFlags);
	UNREFERENCED_PARAMETER(zDelta);
	UNREFERENCED_PARAMETER(pt);

	ReleaseFocus();
	return TRUE;
}

void COSKDlg::OnLButtonUp(UINT nFlags, CPoint point) 
{
	UNREFERENCED_PARAMETER(nFlags);
	UNREFERENCED_PARAMETER(point);

	ReleaseFocus();
}

void COSKDlg::ReleaseFocus() 
{
	/* if(IsWindow(gFocus))
	{
		HWND wnd = ::GetForegroundWindow();
		if(IsWindow(wnd))
		{
			if(wnd == gFocus)
			{
				return;
			}
		}
		::SetForegroundWindow(gFocus);
		::SetFocus(gFocus);
	} */

	HWND hWnd = ::GetWindow(this->m_hWnd, GW_HWNDFIRST);
	LONG lStyle;
	WINDOWPLACEMENT wp;

	ZeroMemory(&wp, sizeof(WINDOWPLACEMENT));
	wp.length = sizeof(WINDOWPLACEMENT);

	while(1)
	{
		if(hWnd != this->m_hWnd)
		{
			lStyle = ::GetWindowLong(hWnd, GWL_STYLE);
			::GetWindowPlacement(hWnd, &wp);

			if(((lStyle & WS_VISIBLE) == WS_VISIBLE) && (wp.showCmd != SW_SHOWMINIMIZED))
			{
				TCHAR tszWindowBuf[256];
				::GetWindowText(hWnd, tszWindowBuf, 255);

				if(tszWindowBuf[0] != 0) break;
			}
		}

		hWnd = ::GetWindow(hWnd, GW_HWNDNEXT);
		if(hWnd == NULL) break;
	}

	if(hWnd != NULL) ::SetForegroundWindow(hWnd);
}

void COSKDlg::OnSize(UINT nType, int cx, int cy) 
{
	CDialog::OnSize(nType, cx, cy);
	if(IsWindow(m_hWnd))
	{
		if(cKeys.GetCount() != 0)
		{
			CalcWidthHeight();
			Invalidate();
		}
	}
}

int COSKDlg::DescribeKeyState() 
{
	int nState = 0;
	short ks;
	ks = GetKeyState(VK_CAPITAL);
	if(ks & 0x000F) nState += 0x01;
	ks = GetKeyState(VK_SHIFT);
	if(ks & 0xF000) nState += 0x02;
	ks = GetKeyState(VK_CONTROL);
	if(ks & 0xF000) nState += 0x04;
	return nState;
}

void COSKDlg::DrawKeyboard()
{
	CClientDC dc(this);
	CRect client;
	CRgn clipRgn;

	GetClientRect(client);
	clipRgn.CreateRectRgnIndirect(&client);

	dc.SelectClipRgn(&clipRgn);
	dc.SelectObject(GetStockObject(DEFAULT_GUI_FONT));
	dc.SetBkMode(TRANSPARENT);

	int x = 0;
	int y = -nKeyHeight;
	KEYDEF * pKey;
	POSITION pos = cKeys.GetHeadPosition();

	cCurrentKeyState = DescribeKeyState();
	int nState = cCurrentKeyState;

	while(pos)
	{
		pKey = cKeys.GetNext(pos);
		if((pKey->cNormal == 0x00) && (pKey->cShifted == 0x00))
		{
			x = 0;
			y += nKeyHeight;
		}
		else
		{
			int width = ((abs(pKey->cWidth) * nKeyWidth) / 10);
			CRect rc(x,y,x + width, y + nKeyHeight);
			x += width;
			if(pKey->cWidth < 0)
			{
				rc.right = client.right; 
			}
			BOOL bHilight = FALSE;
			if(pKey->cNormal == 0x00)
			{
				switch(pKey->cShifted)
				{
				case OSK_CAPSLOCK:
					if(nState & 0x01)
					{
						bHilight = TRUE;
					}
					break;
				case OSK_SHIFT:
					if(nState & 0x02)
					{
						bHilight = TRUE;
					}
					break;
				case OSK_CTRL:
					if(nState & 0x04)
					{
						bHilight = TRUE;
					}
					break;
				default:
					break;
				}
			}
			DrawKey(&dc, rc, pKey, bHilight);
		}
	}
}

void COSKDlg::OnTimer(UINT nIDEvent) 
{
	if(nIDEvent == OSK_TIMER_ID)
	{
		int nState = DescribeKeyState();
		if(nState != cCurrentKeyState) DrawKeyboard();
	}
}
